#! /usr/bin/env python
#
#  a simple undirected Graph class
# 
#

import math
import sys
import copy

class Graph:

    def __init__( s ):
        # default representation in vertex, edge list
        s . V = [] # vertices
        s . E = [] # edges - can be weighted
        s . M = {} # adj matirx - if needed
        s . n = 0
        s . undirected = 0
        s . INFINITY = 99999

    def addVertex( s, v ) :
        if v in s . V:
            print "duplicate Vertex - ignoring", v
        s . V . append ( v )
        s . V . sort()
        
    def rmVertex( s, rv ) :
        for i in range ( len (s . V) - 1, -1, -1):
            if s.V[i] == rv :
                del s.V[ i ]
        for i in range( len( s. E ) - 1, -1, -1) :
            ea, eb, ew = s . E[i]
            if ( ea == rv or eb == rv ):
                del s.E[i]

    def rmEdge( s, va, vb ) :
        if va == vb:
            return
        for i in range( len(s.E) - 1, -1, -1 ) :
            ea, eb, ew = s . E[i]
            if ( va == ea and vb == eb ) :
                del s.E[i]
            if s . undirected:
                if ( va == eb and vb == ea ) :
                    del s.E[i]
            
    def addEdge( s, va, vb, vw = 0, createVertices = 0 ):
        if va not in s .V:
            if createVertices:
                s . addVertex( va )
            else:
                raise VertexNotFound
        if vb not in s .V:
            if createVertices:
                s . addVertex( vb )
            else:
                raise VertexNotFound
        # note: overwrites edge if present
        s . rmEdge ( va, vb ) 
        s . E . append ( ( va, vb, vw ) )
        s . E . sort()

    def degreeOfVertex( s, v ):
        # return degree of input vertex
        degree = 0
        for ea, eb, ew in s . E:
            if ( ea == v or eb == v ) :
                degree = degree + 1
        return degree

    def minDegree( s ):
        # returns minimum degree of entire graph
        minDegree = 999999999
        for v in s . V:
            mind = s.degreeOfVertex( v )
            if mind < minDegree:
                minDegree = mind
        return minDegree

    def addZeroPathVertex( s, zv ):
        s . addVertex( zv )
        for v in s . V:
            if v <> zv :
                s . addEdge( v, zv, 0, 1 )
                if not s . undirected:
                    s . addEdge( zv, v, 0, 1 )
                

    def maxWeight ( s  ):
        # return maximum weight of graph
        mw = 0.0;
        for ea, eb, ew in s . E:
            if ew > mw:
                mw = ew
        return mw

    def totalWeight( s ):
	cost = 0
	for ea, eb, ew in s . E:
	   cost = cost + ew
	return cost

    def nextVert ( s, i ) :
        for ea, eb, ew in s . E:
            if ea == i:
                return eb
        return -1

    def convertEdgePathtoVertList ( s, convertToInt = 0 ):
        ea, eb, ew = s . E[0]
        start = s . track ( ea, 1 )
        nv = s . nextVert ( start )
        vl = []
        vl . append ( start )
        while nv >= 0:
            vl . append ( nv )
            nv = s . nextVert ( nv )
	if convertToInt:
	    # careful - make sure your vertices names are integer-able
	    ivl = []
	    for v in vl:
	    	ivl . append( int( v ))
	    return ivl
        return vl

    def productEdgeWeights( s ):
        # return product of all edge weights
        product = 1
	for ea, eb, ew in s . E:
	   product = product * ew
	return product

    def track( s, i, backward = 0 ) :
        #  follows edges to end
        # NOTE: careful, could be infinite loop
        retVal = i
        for ea, eb, ew in s . E:
            if backward:
                if eb == i :
                    retVal = s . track ( ea, backward )                    
            else:
                if ea == i :
                    retVal =  s . track ( eb )
        return retVal

    def adjustWeights( s, w ):
        for i in range ( len ( s . E )):
            ea, eb, ew = s . E[i]
            s . E[i] = ( ea, eb, ew + w )

    def __str__( s ) :
        s . E . sort()
        s . V . sort()
        rString = "Vertices :" + str( s . V) 
        rString = rString + "\nEdges: "
        for e in s . E:
            va, vb, w = e
            rString = rString + str( va ) + "_" + str( vb ) +'=' + str(w) + " " 
        return rString

    def dump( s, outs ):

        outs . write ( str( len ( s . V )) + ' ' + str( len ( s . E )) + '\n' )
        for v in s . V:
            vString = "%s\n" % str ( v )
            outs . write ( vString )
        for e in s . E:
            va, vb, w = e
            eString = "%s %s %d\n" % ( str(va), str(vb), w )
            outs. write ( eString )

    def saveToFile( s, fileName ):
        outf = open ( fileName, 'w' )
        s . dump ( outf )
        outf . close()

    def saveToSketch( s, fileName, showWeights = 0):
        # write out file in sketch format for pretty render of graph
        outf = open ( fileName, 'w' )

        # header
        outf . write ( """##Sketch 1 2
document()
layout('letter',0)
layer('Layer 1',1,1,0,0,(0,0,0))
""" )
        angleSep = math . pi * 2. / len ( s.V )
        paperWidth = 612
        paperHeight = 792
        graphRadius = 100
        dotRadius = 4
        vPos = {}
        txtOffset = 20

        # vertices
        s . V . sort()
        for i in range( len (s . V)):
            outf . write ( "fp((0,0,0))\n" )
            outf . write ( "lw(1)\n" )
            x = math. cos ( angleSep * i)
            y = math. sin ( angleSep * i)
            px = paperWidth / 2 + graphRadius * x
            py = paperHeight / 2 + graphRadius * y
            vPos[ s. V[i] ] = (px, py )
            outf . write ( "e(" + str(dotRadius) + ",0,0," + str(-dotRadius) + \
                           "," + str(px) + "," + str(py)+ ")\n")
            

        # edges
        for i in range( len( s. E )):
            outf . write( "fp((0,0,0))\n")
            outf . write( "lw(1)\n")
            va, vb, w = s.E[i]
            vax, vay = vPos[va]
            vbx, vby = vPos[vb]
            if not s . undirected:
                outf . write( "la2(([(-4.0, 3.0), (2.0, 0.0), (-4.0, -3.0), (-4.0, 3.0)], 1))\n" )
            outf . write( "b()\n" )
            dy = vby - vay
            dx = vbx - vax
            theta = math . atan2 ( dy, dx )
            dx = math. cos( theta) * 1.5
            dy = math .sin( theta) * 1.5
            outf . write( "bs(" + str(vax) + "," + str(vay) + ",0)\n" )
            outf . write( "bs(" + str(vbx-dotRadius*dx) + "," + str(vby-dotRadius*dy) + ",0)\n" )
            if showWeights:
                if s . undirected:
                    vcx = vax + ( vbx - vax ) / 2.
                    vcy = vay + ( vby - vay ) / 2.
                else:
                    vcx = vax + ( vbx - vax ) / 4.
                    vcy = vay + ( vby - vay ) / 4.
                outf . write ( "fp((0,0,0))\n" )
                outf . write ( "le()\n" )
                outf . write ( "lw(1)\n" )
                outf . write ( "Fn('Times-Roman')\n" )
                dy = vby - vay
                dx = vbx - vax
                theta = math . atan2 ( dy, dx )
                theta = theta + math.pi/2.
                dx = math. cos( theta)
                dy = math .sin( theta)
                tx  = vcx + txtOffset * dx /2. 
                ty =  vcy + txtOffset * dy /2.
                outf . write ( "txt('" +  str(w)+ "',(" + str(tx) + "," + str(ty) + "))\n" )

        # labels
        for i in range( len ( s . V)):
            outf . write ( "fp((0,0,0))\n" )
            outf . write ( "le()\n" )
            outf . write ( "lw(1)\n" )
            outf . write ( "Fn('Times-Roman')\n" )
            vx, vy = vPos[s.V[i]]
            x = math.cos ( angleSep * i )
            y = math.sin( angleSep * i )
            tx = ( graphRadius + txtOffset )* x
            ty = ( graphRadius + txtOffset ) * y
            tx = tx - len ( str( s.V[i] )) * 3
            ty = ty - 3
            tx = paperWidth/2 + tx
            ty = paperHeight/2 + ty
            outf . write ( "txt('" +  str(s.V[i])+ "',(" + str(tx) + "," + str(ty) + "))\n" )
        outf.close()
        

    # matrix representation routines
    # -----------------------------------------
    def buildAdjMatrix ( s ) :        
        s . M . clear()
        for i in range( len(s.V) ):
            for j in range ( len(s.V) ):
                s . M [ i, j ] =  s. INFINITY
                
        s . V . sort()
        for ea, eb, w in s . E:
            aindex = s . V .index( ea )
            bindex = s . V . index( eb )
            s . M[ aindex, bindex ] = w
                        
        # make symmetric
        # this is an undirected graph
        if s . undirected:
            for i in range( len(s.V) ):
                for j in range ( len(s.V) ):
                    if ( s . M . has_key ( (i, j ) )):
                        if s . M[i, j] < s.INFINITY and \
                               s . M[i, j] < s.INFINITY and \
                               s . M[i, j] <> s . M[i,j]:
                            print "graph is not symmetric (undirected) - please check edge", i, j
                            sys . exit(0)
                    elif s . M[j, i] == s.INFINITY:
                        s . M[ j, i ] = s . M[ i, j]

    def numValidEdges( s ) :
        validEdges = 0
        for  edge in s . M . keys():
            if s.M[edge] < s. INFINITY :
                validEdges = validEdges + 1
        return validEdges

    def numRows ( s ) :
        rows = {}
        for  edge in s . M . keys():
            a, b = edge
            rows[a] = 1
        return len ( rows . keys() )

    def deleteRow( s, i ) :
        for ii in range ( len(s.V) ):
            if s  . M . has_key ( ( i, ii )):
                del s . M[ i, ii ]
                
    def deleteCol( s, j ) :
        for jj in range ( len(s.V) ):
            if s . M. has_key( ( jj, j )):
                del s . M[ jj, j ]

    def subtractRow( s, i, v ):
        for ii in range ( len(s.V) ):
            if s . M . has_key( ( i, ii )) and \
                   s . M[i, ii] < s.INFINITY:
                s . M[i, ii] = s . M[i, ii] - v
                
    def subtractCol( s, j, v ):
        for jj in range ( len(s.V) ):
            if s . M . has_key( ( jj,j )) and \
                   s . M[jj, j] < s.INFINITY:
                s . M[jj, j] = s . M[jj, j] - v

    def minRowElement( s,i, jexclude = -1 ) :
        minr = s.INFINITY
        for ii in range ( len(s.V) ):
            if s . M . has_key( (i, ii) ) and s . M[ i, ii ] < minr:
                if ii <> jexclude:
                    minr = s . M [ i, ii ]
        return minr
    
    def minColElement( s,j, iexclude = -1 ) :
        minc = s.INFINITY
        for jj in range ( len(s.V) ):
            if s . M . has_key( (jj, j) ) and s . M[ jj, j ] < minc:
                if jj <> iexclude:
                    minc = s . M [ jj, j ]
        return minc
            
    def reduce ( s ) :
        rValue = 0
        for i in range( len(s.V) ):
            rowred = s . minRowElement( i )
            if rowred < s . INFINITY and rowred > 0:
                s . subtractRow( i, rowred )
                rValue = rValue + rowred
        for i in range( len(s.V) ):
            colred = s . minColElement( i )
            if colred < s.INFINITY and colred > 0:
                s . subtractCol( i, colred )
                rValue = rValue + colred
        return rValue

    def zeroPerRowCol ( s ) :
        # check to make sure one zero in each row
        deadEnd = 0
        for i in range ( len(s.V) ):
            zeroFound = 0
            validRow = 0
            for j in range( len(s.V) ) :
                if s . M. has_key( (i,j)) :
                    validRow = 1
                    if s . M[i,j ] == 0.:
                        if not zeroFound:
                            zeroFound = 1
                        else:
                            deadEnd = 1
            if validRow and not zeroFound:
                deadEnd = 1
        #check to make sure one zero in each col
        for j in range ( len(s.V) ):
            zeroFound = 0
            validCol = 0
            for i in range( len(s.V) ) :
                if s . M. has_key( (i,j)) :
                    validCol = 1
                    if s . M[i,j ] == 0.:
                        if not zeroFound:
                            zeroFound = 1
                        else:
                            deadEnd = 1
            if validCol and not zeroFound:
                deadEnd = 1
        return not deadEnd

                
    def splitEdge( s ):
        score = -s.INFINITY
        bestr = bestc = s.INFINITY
        for i in range ( len(s.V) ):
            for j in range( len(s.V) ) :
                if s . M. has_key( (i,j)) and s . M[i, j] == 0 :
                    minr = s . minRowElement( i, j )
                    minc = s . minColElement( j, i )
                    if minr < s. INFINITY and \
                           minc <  s.INFINITY and \
                               ( minr + minc ) > score:
                        score = minr + minc
                        bestr = i
                        bestc = j

        if bestr == s.INFINITY:
            if s. zeroPerRowCol():
                # just grab first zero off
                for i in range ( len(s.V) ):
                    for j in range( len(s.V) ) :
                        if s . M. has_key( (i,j)) and s . M[i, j] == 0 :
                            bestr = i
                            bestc = j
                            score = 0
        return bestr, bestc, score
                    
    def matrix ( s ):
        strg = "%d %d\n" % ( len(s.V), len(s.V))
        for i in range( len(s.V) ):
            for j in range( len(s.V) ):
                if not s . M . has_key( (i, j ) ):
                    strg = strg + " __ "
                elif s . M[i,j] == s.INFINITY:
                    strg = strg + " xx "
                else:
                    strg = strg + " %02d " % ( s . M[i,j] )
            strg = strg + "\n"
        return strg
                

class TSP :
    
    def __init__ ( s ) :
        s . tweight = 0
        s . tours = []
        s . verbose = 0
        s . optimal = 1
        s . justOne = 0
        s . numVerts = 0
        s . undirected = 0
        s . maxTours = 999999

    def reset ( s ) :
        del s . tours[:]
        s . optimal = 1
        s . justOne = 0
        s . tweight = 0

    def preventCycle ( s, path, a, b ) :
        # find the current path's endpoints
        scb = path . track( a, 1 )  # backward
        sca = path . track( b )     # forward
        return sca, scb

    def branchBound( s, A, cost, tour, rightMost = 0 ):
        cost = cost + A . reduce ( )
        if s . verbose:
            print "branchBound", cost
            print A . matrix()
        cutoff = s . tweight
        if cost < cutoff :
            if A . numRows() == 2 and A . zeroPerRowCol( ): 
                if s . optimal :
                    # adjust goal
                    s . tweight = cost
                # add edges
                for i in range( len(A.V) ):
                    for j in range( len(A.V) ):
                        if A . M. has_key ( (i,j)) and \
                               A . M[i,j] <> A.INFINITY:
                            tour . addEdge( i, j, 0, 1 )
                if len ( tour.E ) != s . numVerts :
                     print "BAD TOUR!!!"
                     print tour
                s . tours . append ( copy.deepcopy(tour) )
                if s . justOne or len( s . tours ) > s.maxTours:
                    s . tweight = 0
                return 
            else:
                bestr, bestc, most = A . splitEdge ( )
                if s . verbose:
                    if most == -A.INFINITY:
                        print "DEAD END"
                    else:
                        print "SPLIT", bestr, bestc, most
                rtour = copy.deepcopy( tour )
                if most > -A.INFINITY:
                    lowerbound = cost + most
                    if bestr < A . INFINITY and \
                       bestc < A . INFINITY :
                        tour . addEdge( bestr, bestc, 0, 1 )
                        sca, scb = s . preventCycle ( tour, bestr, bestc )
                        newA = copy.deepcopy( A )
                        if newA . M. has_key( ( sca, scb ) ) :
                            newA . M[ sca, scb ] = A. INFINITY                             
                        newA . deleteRow ( bestr )
                        newA . deleteCol ( bestc )
                        if s . verbose:
                            print "left ---------------------- with ", bestr, bestc
                            print tour
                        s . branchBound( newA, cost, copy.deepcopy(tour) )
                    if lowerbound < cutoff :
                        # right sub-tree
                        AA = copy.deepcopy( A ) 
                        AA . M [ bestr, bestc ] = AA . INFINITY
                        # prevents reverse paths for undirected graphs
                        if s . undirected and AA . M . has_key ( ( bestc, bestr) ):
                            if rightMost:
                                AA . M [ bestc, bestr ] = newA . INFINITY
                        if s .verbose :
                            print " -------------------------  right w/o", bestr, bestc
                            print rtour
			s . branchBound( AA, cost, rtour, rightMost )
                    else:
                        if s . verbose:
                            print "skip right", lowerbound, s . tweight

    def findMinTour( s, A, weight = -1 ):
        # find all tours <= weight 
        # if weight undefined return an optimal tours
        tour = Graph()
        s . optimal = 1
        if weight == -1:
            s . tweight = A . INFINITY
        else :
            s . tweight = weight
            s . optimal = 0
        AA = copy.deepcopy( A )
        AA . buildAdjMatrix()
        s . numVerts = len ( AA . V ) 
        s . branchBound( AA, 0, tour, 1 )

        # assign names and weights to tours
        A . buildAdjMatrix()
        for t in s.tours:
            for i in range( len( t.E )):
                ea, eb, ew = t.E[i]
                t . E[i] = ( ea, eb, A . M[ea, eb] )
            del t .V[:]
            t. V = A.V[:]
            for i in range( len( t.E )):
                ea, eb, ew = t.E[i]
                t . E[i] = ( t.V[ea], t.V[eb], ew )

        if s . optimal and len ( s . tours ):
            # prune tours
            bestTourLength = A. INFINITY
            for i in range( len(s.tours) ):
                if s . tours[i].totalWeight() < bestTourLength:
                    bestTourLength = s . tours[ i ].totalWeight()
            for i in range( len(s.tours)-1, -1, -1):
                if s.tours[i].totalWeight() > bestTourLength:
                    del s.tours[i]

    def findMaxTour ( s, A, weight = -1 ) :
        # convert to minimum tour by
        # subtracting maxEdge
        maxWeight = A . maxWeight()
        B = copy.deepcopy(A)
        for i in range( len ( B.E )):
            a, b, w = B.E[i]
            B.E[i] = (a, b, maxWeight - w)

        s . findMinTour ( B, weight )      
            
        # restore original weights
        for t in s . tours:
            for i in range( len( t.E )):
                a, b, w = t.E[i]
                t.E[i] = ( a, b, maxWeight - w )

    def findMinPath ( s, A, weight = -1 ):
        # find all paths <= weight by conversion to TSP
        # if weight undefined return optimal path

        # add false edges
        B = copy.deepcopy(A)
        B . addZeroPathVertex ( "ZERO" )

        s . findMinTour ( B, weight )

        for t in s . tours :
            t . rmVertex ( "ZERO" )
       
    def findMaxPath ( s, A, weight = -1 ):

        # convert to minimum path by
        # subtracting maxEdge
        maxWeight = A . maxWeight()
        B = copy.deepcopy(A)
        for i in range( len ( B.E )):
            a, b, w = B.E[i]
            B.E[i] = (a, b, maxWeight - w)
            
        s . findMinPath ( B, weight )

        # restore original weights
        for t in s . tours:
            for i in range( len( t.E )):
                a, b, w = t.E[i]
                t.E[i] = ( a, b, maxWeight - w )
        

if __name__ == '__main__' :


    # builds sample graph and
    # performs a TSP
    a = Graph()

    numVertices = 8
    
    for i in range ( numVertices ):
        a . addVertex ( i )

    booj = 1
    if booj:
        a . addEdge ( 0, 1, 2 ) ;
        a . addEdge ( 0, 2, 11 ) ;
        a . addEdge ( 0, 3, 10 ) ;
        a . addEdge ( 0, 4, 8 ) ;
        a . addEdge ( 0, 5, 7 ) ;
        a . addEdge ( 0, 6, 6 ) ;
        a . addEdge ( 0, 7, 5 ) ;
        
        a . addEdge ( 1, 2, 1 ) ;
        a . addEdge ( 1, 3, 8 ) ;
        a . addEdge ( 1, 4, 8 ) ;
        a . addEdge ( 1, 5, 4 ) ;
        a . addEdge ( 1, 6, 6 ) ;
        a . addEdge ( 1, 7, 7 ) ;
        
        a . addEdge ( 2, 3, 11 ) ;
        a . addEdge ( 2, 4, 8 ) ;
        a . addEdge ( 2, 5, 12 ) ;
        a . addEdge ( 2, 6, 3 ) ;
        a . addEdge ( 2, 7, 11 ) ;
        
        a . addEdge ( 3, 4, 1 ) ;
        a . addEdge ( 3, 5, 9 ) ;
        a . addEdge ( 3, 6, 8 ) ;
        a . addEdge ( 3, 7, 10 ) ;
        
        a . addEdge ( 4, 5, 2 ) ;
        a . addEdge ( 4, 6, 10 ) ;
        a . addEdge ( 4, 7, 9 ) ;
        
        a . addEdge ( 5, 6, 11 ) ;
        a . addEdge ( 5, 7, 9 ) ;

        a . addEdge ( 6, 7, 3 ) ;

        a . addEdge ( 1, 0, 6  ) ;
        a . addEdge ( 2, 0, 5  ) ;
        a . addEdge ( 3, 0, 11 ) ;
        a . addEdge ( 4, 0, 11 ) ;
        a . addEdge ( 5, 0, 12 ) ;
        a . addEdge ( 6, 0, 10 ) ;
        a . addEdge ( 7, 0, 7  ) ;
        a . addEdge ( 2, 1, 12 ) ;
        a . addEdge ( 3, 1, 9  ) ;
        a . addEdge ( 4, 1, 11 ) ;
        a . addEdge ( 5, 1, 8  ) ;
        a . addEdge ( 6, 1, 11 ) ;
        a . addEdge ( 7, 1, 10 ) ;
        a . addEdge ( 3, 2, 10 ) ;
        a . addEdge ( 4, 2, 9  ) ;
        a . addEdge ( 5, 2, 5  ) ;
        a . addEdge ( 6, 2, 12 ) ;
        a . addEdge ( 7, 2, 10 ) ;
        a . addEdge ( 4, 3, 4  ) ;
        a . addEdge ( 5, 3, 2  ) ;
        a . addEdge ( 6, 3, 10 ) ;
        a . addEdge ( 7, 3, 10 ) ;
        a . addEdge ( 5, 4, 11 ) ;
        a . addEdge ( 6, 4, 9  ) ;
        a . addEdge ( 7, 4, 6  ) ;
        a . addEdge ( 6, 5, 12 ) ;
        a . addEdge ( 7, 5, 3  ) ;
        a . addEdge ( 7, 6, 1  ) ;

    else:
      for i in range( len ( a . V )):
        for j in range( len( a . V)):
            if i <> j :
                a . addEdge ( i, j, i + 2*j )
                if i == 0 :
                    a . addEdge ( i, j, i + j )
                if  j == 2:
                    a  . addEdge ( i, j, 4 )

        
    tsp = TSP ( )
    tsp . verbose = 0


    a . saveToSketch( "graph.sk", 1 )

    if not booj:
      for k in range( 100 ):
        tsp . reset()
        tsp . justOne = 0
        print "MIN TOURS", k
        tsp . findMinTour( a, k )
        print len ( tsp.tours )
        for i in range( len( tsp.tours)):
            tName = "mintour" + str(i) + ".sk"
            t = tsp.tours[i]
            t . saveToSketch( tName, 1 )
            print t, t.totalWeight()

    else:
     
        tsp . reset()
        tsp . justOne = 0
        print "MIN TOURS"
        tsp . findMinTour( a )
        for i in range( len( tsp.tours)):
            tName = "mintour" + str(i) + ".sk"
            t = tsp.tours[i]
            # t . saveToSketch( tName, 1 )
            print i, t, t.totalWeight()


        tsp . reset()
        print "MAX TOURS"
        tsp . findMaxTour( a )
        for i in range( len( tsp.tours)):
            tName = "maxtour" + str(i) + ".sk"
            t = tsp.tours[i]
            # t . saveToSketch( tName, 1 )
            print i, t, t.totalWeight()

        tsp . reset()
        print "MIN PATHS"
        tsp . findMinPath( a )
        for i in range( len( tsp.tours)):
            tName = "minpath" + str(i) + ".sk"
            t = tsp.tours[i]
            # t . saveToSketch( tName, 1 )
            print i, t, t.totalWeight()

        tsp . reset()
        print "MAX PATHS"
        tsp . findMaxPath( a )
        for i in range( len( tsp.tours)):
            tName = "maxpath" + str(i) + ".sk"
            t = tsp.tours[i]
            # t . saveToSketch( tName, 1 )
        print i, t, t.totalWeight()

        
