#!  /usr/bin/env python
#
# hbPot.py
#
# An implemntation on the Hydrogen Bond energy potential for
# as described in :
#
# "An imprived hydrogen bond potential: Impact on medium resolution
#  protein strctures"
# F. Fabiola, R. Bertram, A. Korostelev, and M. Chapman
# Protein Science, 11, 1415-1423, 2002.
#

import sys
import trace
import re
import xplorNIH
import xplor
import math
from derivList import DerivList
from pyPot import PyPot
import vec3
from FSUVector import Vector
from FSUMatrix import Matrix
from FSUNMRBase import NMRBase
from atomSel import AtomSel
import copy

class HydrogenBondPot(PyPot):

  def __init__( s, name ):
      PyPot.__init__( s, name, s )
      s . tolerance = 0.5
      s . RadtoDeg = 180. / math . pi
      s . DegtoRad = math . pi / 180.
      
      # These values are explained in the paper, p. 1418

      s . Dcut = 7.5
      s . Doff = 6.5
      s . Don = 5.5
      s . Acut = 100.0
      s . Aoff = 80.0
      s . Aon = 60.0

      # switch settings
      s . Ron = 3.4
      s . Roff = 3.5

      s . thetaLowTetra = 109
      # bimodal
      s . thetaLowTri = 115*s.DegtoRad
      s . thetaHighTri = 155*s.DegtoRad
      s . cosMidTheta = math . cos( (s.thetaLowTri + s.thetaHighTri )/2. )
   
      s . Aexp = 4
      s . Rexp = 6
      s . Angexp = 4
      s . HOcut = 2.7
      s . Rmin = 2.9
      s . sigma = s . Rmin / math . sqrt( 1.5 )

      s . weight = 1.0
      s . cachedHBondList = 0
      s . donorTypes = [ 'N', 'NK', 'NR1', 'NR2', 'NR3', 'NH1', 'NH2',
			 'NN', 'NQ', 'NW', 'OS', 'OT', 'OY', 'OHH' ]
      s . donorHTypes = [ 'H', 'HK1', 'HK2', 'HK3', 'HR1', 'HR2', 'HR3',
			  'HR4', 'HR5', 'HH1', 'HH2', 'HN1', 'HN2', 'HQ1',
			  'HQ2', 'HW', 'HS', 'HT', 'HY', 'HOH1', 'HOH2' ]
      s . acceptorTypes = [ 'O', 'OD1', 'OD2', 'OE1', 'OE2', 'ON', 'OQ' ]
      s . carbonTypes = [ 'C', 'CE', 'CD', 'CT', 'CS', 'CY', 'CQ', 'CN' ]
      s . don_acc = {}
      
     
      s . HBList = []
      s . reset()
      
  def setWeighting ( s, w ) :
    s . weight = w

  def printSettings (s):
    print "----- hydrogen bond-list options -----------"
    print " tolerance = ",s .  tolerance
    print " D(istance)CUToff=", s . Dcut, "DOFF=", s. Doff, "DON=", s.Don 
    print " A(ngle)CUToff=", s . Acut, "AOFF=", s. Aoff, "AON=", s.Aon 
    print " REXP=", s . Rexp, "AEXP=", s . Aexp, "Angleexp=", s . Angexp
    print "---------------------------------------------"

  def reset ( s ):
    s . constructLists( s . donorTypes )
    s . constructLists( s . donorHTypes )
    s . constructLists( s . acceptorTypes )
    s . constructLists( s . carbonTypes )
    s . cosAngCut = math . cos ( s . Acut * s . DegtoRad )
    s . cachedHBondList = 0
    del s . HBList [ : ]

  def isTetrahedral( s, da, db, aa, ab ) :
    tetDonors = [ ('NK',  'HK1' ),  ('NK',  'HK2' ),
                  ('NK' , 'HK3' ),  ('OS',  'HS'  ),
                  ('OT',  'HT'	 ), ('OHH' , 'HOH1' ),
                  ('OHH',	'HOH1' )
                  ]

    tetAcceptors = [ ('OS' ,  'CS' ), ('OT' , 'CT'  ),
                     ('OY' , 'CY' ), ('OHH', 'HOH1'),
                     ('OHH', 'HOH2' )
                     ]
    for ta, tb in tetDonors:
      if da == ta and db == tb :
        for tta, ttb in tetAcceptors:
          if aa == tta and ab == ttb :
            return 1
          
    return 0
  
  def constructLists( s, list ):
    for dt in list:
      if s . don_acc . has_key ( dt ) :
	del s . don_acc [ dt ][:]
      else:
	s . don_acc[ dt ] = []

  def outOfPlaneAngle( s, hb ):

    # this compute the out-of-plane angle
    #
    # Ref. set Fig. 22 of
    # "Hydrogen Bonding in Globular Proteins"
    # E.N. Baker and R.E. Hubbard
    # Proc. Biophys. Molec. Biol., vol 44, pp. 97-179, 1984.
    
    N = hb[0]
    O = hb[1]
    H = hb[2]
    C = hb[3]
    
    selString = '( name CA)'
    resNum = C . residueNum( )
    for atom in AtomSel ( selString ):
      if atom . residueNum( ) == resNum:
        CA = atom

    vN = N . pos()
    vO = O . pos()
    vH = H . pos()
    vC = C . pos()
    vCA = CA . pos()

    OH = vH - vO 
    CC = vCA - vC 
    CO = vO - vC 

    OHlen = vec3 . norm( OH )
    CClen = vec3 . norm( CC )
    COlen = vec3 . norm( CO )

    ppN = vec3 . cross( CC, CO )
    ppNlen = vec3 . norm( ppN )
    OHdN = vec3 . dot( OH, ppN ) / ( ppNlen * OHlen )
    ppN = OHdN * ppN
    inPlane = OH - ppN 
    inPlanelen = vec3 .  norm( inPlane ) 
  
    beta = math . acos ( vec3 . dot ( inPlane, OH ) / (inPlanelen * OHlen )) * 180. / math.pi
    gamma = math . acos ( vec3 . dot( CO, inPlane ) / (inPlanelen * COlen )) * 180. / math .pi

    # get correct sign
    pos = vec3 . dot ( CO, CC )
    if pos * gamma < 0. :
      gamma = -1* gamma
      
    return beta, gamma


  def createHBondLists( s, printOut = 0 ):
    # build lists of Hydrogen bonds
    # run through all atoms

    s . reset( ) 

    selString = '(all)'
    allAtoms = AtomSel ( selString )
    for atom in allAtoms :

      # ------------------arrays of donor N and O atoms ----------------------------

      if   ((atom.atomName() ==	  'N') and (atom.chemType() == 'NH1')) :
	# main 'N'
	s . don_acc[ 'N' ] . append ( atom ) 
      elif ((atom.atomName() ==	 'NZ') and (atom.chemType() == 'NH3')) :
	#  Lys 'NZ'
	s . don_acc[ 'NK' ] . append( atom ) 
      elif ((atom.atomName() == 'NH1') and (atom.chemType() == 'NC2')) :
	# Arg 'NH1'
	s . don_acc[ 'NR1' ] . append( atom ) 
      elif ((atom.atomName() == 'NH2') and (atom.chemType() == 'NC2')) :
	# Arg 'NH2'
	s . don_acc[ 'NR2' ] . append( atom ) 
      elif ((atom.atomName() ==	 'NE') and (atom.chemType() == 'NH1')) :
	# Arg 'NE'
	s . don_acc[ 'NR3' ] . append( atom )
      elif ((atom.atomName() == 'ND1') and (atom.chemType() == 'NH1')) :
	# His 'ND1'
	s . don_acc[ 'NH1' ] . append( atom )
      elif ((atom.atomName() == 'NE2') and (atom.chemType() == 'NH1')) :
	# His 'NE2'
	s . don_acc[ 'NH2' ] . append( atom ) 
      elif ((atom.atomName() == 'ND2') and (atom.chemType() == 'NH2')) :
	# Asn 'ND2'
	s . don_acc[ 'NN' ] . append( atom )
      elif ((atom.atomName() == 'NE2') and (atom.chemType() == 'NH2')) :
	# Glu 'NE2'
	s . don_acc[ 'NQ' ] . append( atom )
      elif ((atom.atomName() == 'NE1') and (atom.chemType() == 'NH1')) :
	# Trp 'NE1'
	s . don_acc[ 'NW' ] . append( atom )
      elif ((atom.atomName() ==	 'OG') and (atom.chemType() == 'OH1')) :
	# Ser 'OG'
	s . don_acc[ 'OS' ] . append( atom )
      elif ((atom.atomName() == 'OG1') and (atom.chemType() == 'OH1')) :
	# Thr 'OG1'
	s . don_acc[ 'OT' ] . append( atom )
      elif ((atom.atomName() ==	 'OH') and (atom.chemType() == 'OH1')) :
	# Tyr 'OH'
	s . don_acc[ 'OY' ] . append( atom )
      elif ((atom.atomName() ==	  'O') and (atom.chemType() ==	'OT')) :
	# HOH 'O'
	s . don_acc[ 'OHH' ] . append( atom )

      # ---------------------Arrays of Donor H atoms--------------------------

      if   ((atom.atomName() ==	   'H') and (atom.chemType() ==	 'H')) :
	# amide H.
	s . don_acc[ 'H' ] . append( atom )
      elif ((atom.atomName() ==	  'HN') and (atom.chemType() ==	 'H')) :
	# amide H.
	s . don_acc[ 'H' ] . append( atom )
      elif ((atom.atomName() ==	 'HZ1') and (atom.chemType() == 'HC')) :
	# Lys H.
	s . don_acc[ 'HK1' ] . append( atom )
      elif ((atom.atomName() ==	 'HZ2') and (atom.chemType() == 'HC')) :
	# Lys H.
	s . don_acc[ 'HK2' ] . append( atom )
      elif ((atom.atomName() ==	 'HZ3') and (atom.chemType() == 'HC')) :
	# Lys H.
	s . don_acc[ 'HK3' ] . append( atom )
      elif ((atom.atomName() == 'HH11') and (atom.chemType() == 'HC')) :
	# Arg H.
	s . don_acc[ 'HR1' ] . append( atom )
      elif ((atom.atomName() == 'HH12') and (atom.chemType() == 'HC')) :
	# Arg H.
	s . don_acc[ 'HR2' ] . append( atom )
      elif ((atom.atomName() == 'HH21') and (atom.chemType() == 'HC')) :
	# Arg H.
	s . don_acc[ 'HR3' ] . append( atom )
      elif ((atom.atomName() == 'HH22') and (atom.chemType() == 'HC')) :
	# Arg H.
	s . don_acc[ 'HR4' ] . append( atom )
      elif ((atom.atomName() ==	  'HE') and (atom.chemType() ==	 'H')) :
	# Arg H.
	s . don_acc[ 'HR5' ] . append( atom )
      elif ((atom.atomName() ==	 'HD1') and (atom.chemType() ==	 'H')) :
	# His H.
	s . don_acc[ 'HH1' ] . append( atom )
      elif ((atom.atomName() ==	 'HE2') and (atom.chemType() ==	 'H')) :
	# His H.
	s . don_acc[ 'HH2' ] . append( atom )
      elif ((atom.atomName() == 'HD21') and (atom.chemType() ==	 'H')) :
	# Asn H.
	s . don_acc[ 'HN1' ] . append( atom )
      elif ((atom.atomName() == 'HD22') and (atom.chemType() ==	 'H')) :
	# Asn H.
	s . don_acc[ 'HN2' ] . append( atom )
      elif ((atom.atomName() == 'HE21') and (atom.chemType() ==	 'H')) :
	# Gln H.
	s . don_acc[ 'HQ1' ] . append( atom )
      elif ((atom.atomName() == 'HE22') and (atom.chemType() ==	 'H')) :
	# Gln H.
	s . don_acc[ 'HQ2' ] . append( atom )
      elif ((atom.atomName() ==	 'HE1') and (atom.chemType() ==	 'H')) :
	# Trp H.
	s . don_acc[ 'HW' ] . append( atom )
      elif ((atom.atomName() ==	  'HG') and (atom.chemType() ==	 'H')) :
	# Ser H.
	s . don_acc[ 'HS' ] . append( atom )
      elif ((atom.atomName() ==	 'HG1') and (atom.chemType() ==	 'H')) :
	# Thr H.
	s . don_acc[ 'HT' ] . append( atom )
      elif ((atom.atomName() ==	  'HH') and (atom.chemType() ==	 'H')) :
	# Tyr H.
	s . don_acc[ 'HY' ] . append( atom )
      elif ((atom.atomName() ==	  'H1') and (atom.chemType() == 'HT')) :
	# HOH H.
	s . don_acc[ 'HOH1' ] . append( atom )
      elif ((atom.atomName() ==	  'H2') and (atom.chemType() == 'HT')) :
	# HOH H.
	s . don_acc[ 'HOH2' ] . append( atom )
	pass

      # ---------------------Arrays of Acceptor atoms--------------------------

      if  ((atom.atomName() ==	  'O') and (atom.chemType() ==	'O')) :
	# main Os
	s . don_acc[ 'O' ] . append( atom )
      elif ((atom.atomName() == 'OD1') and (atom.chemType() == 'OC')) :
	# Asp OD1
	s . don_acc[ 'OD1' ] . append( atom )
      elif ((atom.atomName() == 'OD2') and (atom.chemType() == 'OC')) :
	# Asp OD2
	s . don_acc[ 'OD2' ] . append( atom )
      elif ((atom.atomName() == 'OE1') and (atom.chemType() == 'OC')) :
	# Glu OE1
	s . don_acc[ 'OE1' ] . append( atom )
      elif ((atom.atomName() == 'OE2') and (atom.chemType() == 'OC')) :
	# Glu OE2
	s . don_acc[ 'OE2' ] . append( atom )
      elif ((atom.atomName() == 'OD1') and (atom.chemType() ==	'O')) :
	# Asn OD1
	s . don_acc[ 'ON' ] . append( atom )
      elif ((atom.atomName() == 'OE1') and (atom.chemType() ==	'O')) :
	# Gln OE1
	s . don_acc[ 'OQ' ] . append( atom )

      # ---------------------Arrays of Carbon atoms--------------------------

      if   ((atom.atomName() ==	 'C') and (atom.chemType() == 'C')) :
	# main Cs
	 s . don_acc[ 'C' ] . append( atom )
      elif ((atom.atomName() == 'CD') and (atom.chemType() == 'C') and
	    (atom.residueName() == 'GLU')) :
	# Glu CD
	s . don_acc[ 'CE' ] . append( atom )
      elif ((atom.atomName() == 'CG') and (atom.chemType() == 'C') and
	    (atom.residueName() == 'ASP')) :
	# Asp CG
	s . don_acc[ 'CD' ] . append( atom )
      elif ((atom.atomName() == 'CG2') and (atom.chemType() == 'CH3E') and
	    (atom.residueName() == 'THR')) :
	# Thr CG
	s . don_acc[ 'CT' ] . append( atom )
      elif ((atom.atomName() == 'CB') and (atom.chemType() == 'CH2E') and
	    (atom.residueName() == 'SER')) :
	# Ser CB
	s . don_acc[ 'CS' ] . append( atom )
      elif ((atom.atomName() == 'CZ') and (atom.chemType() == 'CY2') and
	    (atom.residueName() == 'TYR')) :
	# Tyr CZ
	s . don_acc[ 'CY' ] . append( atom )
      elif ((atom.atomName() == 'CD') and (atom.chemType() == 'C') and
	    (atom.residueName() == 'GLN')) :
	# Gln CD
	s . don_acc[ 'CQ' ] . append( atom )
      elif ((atom.atomName() == 'CG') and (atom.chemType() == 'C') and
	    (atom.residueName() == 'ASN')) :
	# Asn CG
	s . don_acc[ 'CN' ] . append( atom )

    donorAtoms = [ ('NR1', 'HR1' ), ('NR1', 'HR2' ), ('NR2', 'HR3' ),
		   ('NR2', 'HR4' ), ('NR3', 'HR5' ), ('NH1', 'HH1' ),
		   ('NH2', 'HH2' ), ('NN' , 'HN1' ), ('NN' , 'HN2' ),
		   ('NQ' , 'HQ1' ), ('NQ' , 'HQ2' ), ('NW' , 'HW'  ),
		   ('OY',  'HY'	 ), ('N'  , 'H'	  ), ('NK',  'HK1' ),
		   ('NK',  'HK2' ), ('NK' , 'HK3' ), ('OS',  'HS'  ),
		   ('OT',  'HT'	 ), ('OHH' , 'HOH1' ), ('OHH',	'HOH1' )
		   ]

    acceptorAtoms = [ ('OE1', 'CE'  ), ('OE2', 'CE' ), ('ON' , 'CN' ),
		      ('OQ' , 'CQ'  ), ('OD1', 'CD' ), ('OD2', 'CD' ),
		      ('O'  , 'C'  ), ('OS' ,  'CS' ), ('OT' , 'CT'  ),
		      ('OY' , 'CY' ), ('OHH', 'HOH1'), ('OHH', 'HOH2' )
		      ]

    

    for da, db in donorAtoms:
      donorsA = s . don_acc [ da ] 
      donorsB = s . don_acc [ db ]
      	
      for aa, ab in acceptorAtoms:
	accsA = s . don_acc [ aa ]
	accsB = s . don_acc [ ab ]

        isTetra = s . isTetrahedral ( da, db, aa, ab )

	for di in range( len( donorsA )):
	  N = donorsA[ di ]
	  H = donorsB[ di ]
	  for ai in range( len( accsA )):
	    O = accsA[ ai ]
	    C = accsB[ ai ]

	    if	( N . residueNum() == O . residueNum( ) and
		  N . chemType() == O . chemType ( ) ) :
	      # skip same residue
	      continue

	    vN = N . pos()
	    vO = O . pos()
	    vH = H . pos()
	    vC = C . pos()

	    ON = vN - vO
	    OH = vH - vO
	    NH = vH - vN
	    OC = vC - vO

	    ONlen = vec3 . norm( ON )
	    OHlen = vec3 . norm( OH )
	    NHlen = vec3 . norm( NH )
	    OClen = vec3 . norm( OC )

	    ONdotOC = vec3 . dot( ON, OC )
	    OHdotNH = vec3 . dot( OH, NH )

	    ONcrossOC = vec3 . cross( ON, OC )
	    crossMag = vec3 . norm( ONcrossOC )

	    cosang = ONdotOC / ( ONlen * OClen )
	    sinang = crossMag / ( ONlen * OClen )
	    cosangH = OHdotNH / ( OHlen * NHlen )

	    if cosang < s .cosAngCut :
	      
	      if ( ONlen < s . Dcut and
		   OHlen < s . HOcut ) :

                # add out-of-plane criteria
                oopBeta, oopGamma = s . outOfPlaneAngle(  [ N, O, H, C ] )
                if oopBeta > 50. or oopGamma < -50.:
                  # these values are chosen based on graph in Fig. 24 of
                  # "Hydrogen BOnding in Globular Proteins"
                  # see ::outOfPlane function for reference details
                  continue


		if ( cosangH <= 0. ) :

                  if printOut :
                    
                    print "---- H-bond ---- "
                    print "\tDonor: ", N . residueNum(), N . chemType(), N . residueName()
                    print "\tAcceptor: ", O . residueNum(), O . chemType(), O . residueName()
                    print "\tC residue = ", C . residueNum(), C . chemType(), C . residueName()
                    print "\tangle C-O-N = ", s . RadtoDeg * math . acos ( cosang )
                    print "\tangle N-H-O = ", s . RadtoDeg * math . acos ( cosangH )
                    print "\tO-C distance = ", OClen
                    print "\tO-N distance = ", ONlen
                    print "\tH-O distance = ", OHlen
                    print "\tH-N distance = ", NHlen
                    print "\tEnergy = ", s . calcHBEnergy( [ N, O, H, C, isTetra ] )
	    
                  s . HBList . append ( [ N, O, H, C, isTetra ] )
                  
	    
  def calcEnergy ( s, printOut = 0 ) :

    s . createHBondLists( )

    totE = 0.
    for hb in s . HBList:
      E, vdw, sw, theta, cosDelta = s . calcHBEnergy( hb )
      totE += E
    
    return s . weight * totE


  def calcEnergyAndDerivs(s,derivs):

    s . createHBondLists( 1 )
      
    totE = 0.
    
    #for hb in s . HBList[ 0 ]:
    for i in range( len( s . HBList) ):
      
      hb = s . HBList[ i ]

      N = hb[0]
      O = hb[1]
      H = hb[2]
      C = hb[3]
 
      dO = vec3.Vec3( 0., 0., 0.)
      dN = vec3.Vec3( 0., 0., 0.)
      dC = vec3.Vec3( 0., 0., 0.)

      E, vdw, switch, theta0, cosDelta = s . calcHBEnergy( hb )
      totE += E
      s . calcDerivs ( N, O, H, C, dO, dN, dC, vdw, switch, theta0, cosDelta )
      
      derivs [ O . index() ] = derivs [ O . index() ] + dO
      derivs [ N . index() ] = derivs [ N . index() ] + dN
      derivs [ C . index() ] = derivs [ C . index() ] + dC


    return s . weight * totE
  

  def calcHBEnergy( s, hb ):

    N = hb[0]
    O = hb[1]
    H = hb[2]
    C = hb[3]
    isTetra = hb[4]
    
    vN = N . pos()
    vO = O . pos()
    vH = H . pos()
    vC = C . pos()
    
    ON = vN - vO
    OH = vH - vO
    NH = vH - vN
    OC = vC - vO

    R = ONlen = vec3 . norm( ON )
    OHlen = vec3 . norm( OH )
    OClen = vec3 . norm( OC )
    
    ONdotOC = vec3 . dot( ON, OC )
    ONcrossOC = vec3 . cross( ON, OC )
    crossMag = vec3 . norm( ONcrossOC )
    
    cosang = ONdotOC / ( ONlen * OClen )
    sinang = crossMag / ( ONlen * OClen )
    
    if isTetra:
      # tetrahedral
      theta0 = s . thetaLowTetra 
    else:
      # triagonal
      if cosang > s . cosMidTheta :
        theta0 = s . thetaLowTri
      else:
        theta0 = s . thetaHighTri

    cosDelta = cosang * math.cos( theta0 ) + sinang * math.sin( theta0 )
    vdwTerm = math. pow( s . sigma / R, s. Rexp ) - math . pow( s . sigma/R, s . Aexp )

    # check switching
    if ( R <= s . Ron ) :
      swFactor = 1.0
    elif R > s. Ron and R < s . Roff :
      swFactor =(s.Roff-R)*(s.Roff-R)*(s.Roff+2.0*R-3.0*s.Ron)/math.pow((s.Roff-s.Ron),3) 
    else:
      swFactor = 0.

    E = vdwTerm * math.pow( cosDelta, s . Angexp )
    return E, vdwTerm, swFactor, theta0, cosDelta

  def calcDerivs (s, N, O, H, C, dO, dN, dC, vdw, sw, theta0, cosDelta ):

    vN = N . pos()
    vO = O . pos()
    vH = H . pos()
    vC = C . pos()
    
    ON = vN - vO
    OH = vH - vO
    NH = vH - vN
    OC = vC - vO

    # conversions
    onx = ON[0]
    ony = ON[1]
    onz = ON[2]
    ocx = OC[0]
    ocy = OC[1]
    ocz = OC[2]
    OClen = vec3 . norm( OC )
    OClenSq = OClen * OClen
    OClenCub = OClen * OClen * OClen
    R = ONlen = vec3 . norm( ON )
    ONlenSq = ONlen * ONlen
    ONlenCub = ONlen * ONlen * ONlen

    ONdotOC = vec3 . dot( ON, OC )
    ONcrossOC = vec3 . cross( ON, OC )
    cross = vec3 . norm( ONcrossOC )
    cangle = ONdotOC / ( ONlen * OClen )  

    #	 first compute derivative of the van der Waals-like term.
    
    R3 = math.pow(R, 3 )
    K = s.sigma*(s.Rexp*math . pow( (s.sigma/R), (s.Rexp-1.0)) - \
                 s.Aexp*math . pow( (s.sigma/R), (s.Aexp-1.0))  ) /  R3
    dxovdw = K*onx
    dyovdw = K*ony
    dzovdw = K*onz

    dxnvdw = -dxovdw             
    dynvdw = -dyovdw
    dznvdw = -dzovdw

    dxcvdw = 0.0
    dycvdw = 0.0
    dzcvdw = 0.0

    #	 compute derivative of cos(theta).

    dxoct = -(ocx+onx)/(OClen*ONlen) + (cangle*ocx)/(OClenSq) + (cangle*onx)/(ONlenSq)
    dyoct = -(ocy+ony)/(OClen*ONlen) + (cangle*ocy)/(OClenSq) + (cangle*ony)/(ONlenSq)
    dzoct = -(ocz+onz)/(OClen*ONlen) + (cangle*ocz)/(OClenSq) + (cangle*onz)/(ONlenSq)

    dxnct = ocx/(OClen*ONlen) - (cangle*onx)/(ONlenSq) 
    dynct = ocy/(OClen*ONlen) - (cangle*ony)/(ONlenSq)
    dznct = ocz/(OClen*ONlen) - (cangle*onz)/(ONlenSq)

    dxcct = onx/(OClen*ONlen) - (cangle*ocx)/(OClenSq)
    dycct = ony/(OClen*ONlen) - (cangle*ocy)/(OClenSq)
    dzcct = onz/(OClen*ONlen) - (cangle*ocz)/(OClenSq)
    
    #	  compute derivative of sin(theta). Need to first compute derivatives of
    #	  cross product, etc.
    
    dxocross = ((onz*ocx-onx*ocz)*(ocz-onz) + (onx*ocy-ony*ocx)*(ony-ocy))/cross 
    dyocross = ((ony*ocz-onz*ocy)*(onz-ocz) + (onx*ocy-ony*ocx)*(ocx-onx))/cross 
    dzocross = ((ony*ocz-onz*ocy)*(ocy-ony) + (onz*ocx-onx*ocz)*(onx-ocx))/cross 
    dxncross = (ocy*(onx*ocy-ony*ocx) - ocz*(onz*ocx-onx*ocz))/cross 
    dyncross = (ocz*(ony*ocz-onz*ocy) - ocx*(onx*ocy-ony*ocx))/cross 
    dzncross = (ocx*(onz*ocx-onx*ocz) - ocy*(ony*ocz-onz*ocy))/cross 
    dxccross = (onz*(onz*ocx-onx*ocz) - ony*(onx*ocy-ony*ocx))/cross 
    dyccross = (onx*(onx*ocy-ony*ocx) - onz*(ony*ocz-onz*ocy))/cross 
    dzccross = (ony*(ony*ocz-onz*ocy) - onx*(onz*ocx-onx*ocz))/cross
    
    dxooci = ocx/OClenCub
    dyooci = ocy/OClenCub
    dzooci = ocz/OClenCub
    dxnoci = 0.0
    dynoci = 0.0
    dznoci = 0.0
    dxcoci = -ocx/OClenCub
    dycoci = -ocy/OClenCub
    dzcoci = -ocz/OClenCub
    
    dxooni = onx/ONlenCub
    dyooni = ony/ONlenCub
    dzooni = onz/ONlenCub
    dxnoni = -onx/ONlenCub
    dynoni = -ony/ONlenCub
    dznoni = -onz/ONlenCub
    dxconi = 0.0
    dyconi = 0.0
    dzconi = 0.0
     
    dxost = dxocross/(ONlen*OClen) + cross*dxooci/ONlen + cross*dxooni/OClen
    dyost = dyocross/(ONlen*OClen) + cross*dyooci/ONlen + cross*dyooni/OClen
    dzost = dzocross/(ONlen*OClen) + cross*dzooci/ONlen + cross*dzooni/OClen
    dxnst = dxncross/(ONlen*OClen) + cross*dxnoci/ONlen + cross*dxnoni/OClen
    dynst = dyncross/(ONlen*OClen) + cross*dynoci/ONlen + cross*dynoni/OClen
    dznst = dzncross/(ONlen*OClen) + cross*dznoci/ONlen + cross*dznoni/OClen
    dxcst = dxccross/(ONlen*OClen) + cross*dxcoci/ONlen + cross*dxconi/OClen
    dycst = dyccross/(ONlen*OClen) + cross*dycoci/ONlen + cross*dyconi/OClen
    dzcst = dzccross/(ONlen*OClen) + cross*dzcoci/ONlen + cross*dzconi/OClen

    #    compute derivative of cos(delta theta)**expangl.
    
    K2 = s.Angexp*math . pow( cosDelta, (s.Angexp-1))
    cthetao = math . cos( theta0 )
    sthetao = math . sin( theta0 )
    dxocdt = K2 * (cthetao*dxoct+sthetao*dxost)
    dyocdt = K2 * (cthetao*dyoct+sthetao*dyost)
    dzocdt = K2 * (cthetao*dzoct+sthetao*dzost)
    dxncdt = K2 * (cthetao*dxnct+sthetao*dxnst)
    dyncdt = K2 * (cthetao*dynct+sthetao*dynst)
    dzncdt = K2 * (cthetao*dznct+sthetao*dznst)
    dxccdt = K2 * (cthetao*dxcct+sthetao*dxcst)
    dyccdt = K2 * (cthetao*dycct+sthetao*dycst)
    dzccdt = K2 * (cthetao*dzcct+sthetao*dzcst)
    
    #    compute derivatives of switching function.
    
    if ((R <  s.Ron) or (R > s.Roff)) :
      dxosw  = 0.0
      dyosw  = 0.0
      dzosw  = 0.0
      dxnsw  = 0.0
      dynsw  = 0.0
      dznsw  = 0.0
      dxcsw  = 0.0
      dycsw  = 0.0
      dzcsw  = 0.0
    else:
      RoffRon3  = 1.0/math.pow((s.Roff-s.Ron),3)
      K3 = -2.0*RoffRon3*((s.Roff-R)*(s.Roff-R)-(s.Roff+2.0*R-3.0*s.Ron)*(s.Roff-R))/R
      dxosw  = onx*K3
      dyosw  = ony*K3
      dzosw  = onz*K3
      
      dxnsw  = -dxosw
      dynsw  = -dyosw
      dznsw  = -dzosw
      
      dxcsw  = 0.0
      dycsw  = 0.0
      dzcsw  = 0.0

    #    compute derivatives of H-bond energy.

    Kexp = math . pow( cosDelta, s . Angexp )
    dxoehb = Kexp*sw*dxovdw+ vdw*sw*dxocdt + vdw*Kexp*dxosw
    dyoehb = Kexp*sw*dyovdw+ vdw*sw*dyocdt + vdw*Kexp*dyosw
    dzoehb = Kexp*sw*dzovdw+ vdw*sw*dzocdt + vdw*Kexp*dzosw
    dxnehb = Kexp*sw*dxnvdw+ vdw*sw*dxncdt + vdw*Kexp*dxnsw
    dynehb = Kexp*sw*dynvdw+ vdw*sw*dyncdt + vdw*Kexp*dynsw
    dznehb = Kexp*sw*dznvdw+ vdw*sw*dzncdt + vdw*Kexp*dznsw
    dxcehb = Kexp*sw*dxcvdw+ vdw*sw*dxccdt + vdw*Kexp*dxcsw
    dycehb = Kexp*sw*dycvdw+ vdw*sw*dyccdt + vdw*Kexp*dycsw
    dzcehb = Kexp*sw*dzcvdw+ vdw*sw*dzccdt + vdw*Kexp*dzcsw
    
    #    send derivatives back to XPLOR

    dN[0] = dxnehb
    dN[1] = dynehb
    dN[2] = dznehb
    dO[0] = dxoehb
    dO[1] = dyoehb
    dO[2] = dzoehb
    dC[0] = dxcehb
    dC[1] = dycehb
    dC[2] = dzcehb

  def writePyMol ( s, pyFile ):
    # convenience rountine

    if not s . cachedHBondList:
      s . createHBondLists()

    pf = open ( pyFile, "w" )
      
    for hb in s . HBList:
      N = hb[0]
      O = hb[1]
      H = hb[2]
      C = hb[3]
      
      dStr = "distance " + \
             "(name " +  N . atomName() +  " and resi " + str(N . residueNum()) + \
             "), (name " + O. atomName() + " and resi " +   str(O . residueNum()) + ")"
      pf . write( dStr + '\n' )

    pf . close()

     
if __name__ == '__main__' :

  hb                         = HydrogenBondPot ( "hydrogenBondPot" )

  
