#! /usr/bin/env python
#
# perform very basic ssNMR calculations
#
#

import sys
import math
import getopt
import string
import re
from random import Random 

from FSUVector import Vector
from FSUMatrix import Matrix

class NMRBase :

#
# Computes ssNMR PISEMA data
#
#  Main reference is:
#     "Atomic Refinement Using Orientational Restraints from ssNMR"
#     Bertram, Quine, Chapman and Cross
#     Journal of Magnetic Resonance, 147, 9-16. (2000)
#
#
    def __init__ ( self ) :
        self . sigma11 = 0.
        self . sigma22 = 0.
        self . sigma33 = 0.
        self . nuParallel = 0.
        self . betaRad = 0.
        self . alphaRad = 0.
	# normally compute abs(nu) 
	self . reflect = 1
        # correlation flags
        self . correlateNone = 0
        self . correlate1D = 1
        self . correlate2D = 2
	self . NumErr = 0.001

    def toDegree ( self, x ) :
        return ( x * 180. / math.pi )

    def toRad ( self, x ) :
        return ( x * math.pi / 180. )

    def error ( self, a, b ) :
        return ( ( float ( a ) - float ( b ) ) / float ( b )  * 100. )

    def setBetaDetermination ( self, bb ) :
        self . rawbeta = bb
    
    def getNu ( self, sigma, neg = 0 ) :
	sigmaBar = ( sigma - self . sigma11 ) / ( self . sigma33 - self . sigma11 ) 
        if (sigmaBar < 0. and sigmaBar > -self.NumErr) :
            sigmaBar = 0.
	nuBar = math . cos ( 2. * self . betaRad ) * sigmaBar + \
		math . sin ( self . betaRad ) * math . sin ( self . betaRad ) 
	if neg :
	   nuBar = nuBar - math . sqrt ( sigmaBar * ( 1.0 - sigmaBar ) * \
			math . sin ( 2. * self . betaRad ) *  \
			math . sin ( 2. * self . betaRad ) )
	else :
	   nuBar = nuBar + math . sqrt ( sigmaBar * ( 1.0 - sigmaBar ) * \
			math . sin ( 2. * self . betaRad ) * \
			math . sin ( 2. * self . betaRad ) )
	# perform inverse transform
	if ( nuBar > 1. ) :
		print "NuBar Error ", sigmaBar, nuBar1, nuBar
	nu = self . nuParallel / 2. * ( 3. * nuBar - 1. ) 
	if self . reflect :
		nu = math . fabs( nu )
	return nu

    def getDipolePowder ( self, a, b, c, B ) :

        theta, enu = self . calculateNuAndTheta ( B, a, b )
        pas = self . computePAF ( a, b, c, self . betaRad, self . alphaRad )
        esigma = self . computeChemShift ( pas, B )
                    
        return esigma, enu, theta

    def calculateNuAndTheta ( self, B, a, b ) :
        ab = b . subtract ( a )
	ab . normalize ( )
        theta = B . dot ( ab ) 
        costhetasq = theta * theta 
	nu = self . nuParallel / 2. * ( 3. * costhetasq - 1. )
        theta = math . acos ( theta )
	return ( theta, nu ) 
   
    def computePAF ( self, A, B, C, betaRadians, alphaRadians ) :
        # see Denny pp. 11-12
	f = self . computeF ( A, B, C )
	mbeta = Matrix( ) 
	mbeta . setRotZ( -betaRadians )
	nalpha = Matrix( ) 
	nalpha . setRotX( -alphaRadians )
	fm = f . mult ( mbeta )   
	pas = fm . mult ( nalpha )
	return pas

    def computeF ( self, A, B, C ) :
        u1 = B . subtract ( A )
	u1 . normalize ( ) 
	u2 = C . subtract ( A )
	u2 . normalize ( )
	b = u1 . cross ( u2 )
	b . normalize ()
	n = b . cross ( u1 )
	n . normalize ( )
	F = Matrix ( ) 
	F . setCol ( 0, u1 ) 
	F . setCol ( 1, n )
	F . setCol ( 2, b )
        return F

    def computeChemShift ( self, pas, B, printSigmas = 0, index11 = 1, index22 = 2, index33 = 0 ) :
	sigma11v = pas . getCol ( index11 ) 
	sigma22v=  pas . getCol ( index22 ) 
	sigma33v = pas . getCol ( index33 )
        if printSigmas :
            print "SIGMA 11: ", sigma11v
            print "SIGMA 22: ", sigma22v
            print "SIGMA 33: ", sigma33v
            Bpas = pas . postMultRowVec ( B ) 
            print "B in PAF", Bpas [index11], Bpas[index22], Bpas[index33]
	B11dot = sigma11v . dot ( B ) 
	B22dot = sigma22v . dot ( B ) 
	B33dot = sigma33v . dot ( B )
	return ( self . sigma11 * B11dot * B11dot + \
		 self . sigma22 * B22dot * B22dot + \
		 self . sigma33 * B33dot * B33dot )

    def getTrueBinPAF ( self, pas, B, index11 = 1, index22 = 2, index33 = 0  ) :
         Bpas = pas . postMultRowVec ( B )
         trueB = Vector ( Bpas[index11], Bpas[index22], Bpas[index33] )
         return trueB

    def determineDipoleSign ( s, nu, sigma, correlation ) :
        if correlation == s . correlateNone:
            return 0
        elif correlation == s . correlate1D:
            if nu >= s .nuParallel / 2. :
                return 1
            else:
                return 0
        elif correlation == s . correlate2D:
            # 2D correlation
            sigmaBar = ( sigma - s . sigma11 ) / ( s . sigma33 - s . sigma11 )
            if (sigmaBar < 0. and sigmaBar > -0.0001 ) :
                sigmaBar = 0.
            nuPos = s . getNu ( sigma )
            nuNeg = s . getNu ( sigma, -1 )
            ellipse = 0
            if nu >= nuNeg and nu <= nuPos:
                ellipse = ellipse + 1
            if nu >= -nuPos and nu <= -nuNeg:
                ellipse = ellipse - 1
            return ellipse
            
    def getCorrelatedError ( s, calc, obs, sigma, correlate ) :
        # this return the correlated dipole
        # error based on looking at the corresponding cs
        # values and determining where on the freq plane
        # the data point
        sgn = s . determineDipoleSign ( obs , sigma, correlate )
        if sgn == 0:
            delta = math . fabs ( calc ) - obs
        elif sgn == -1:
            delta = calc + obs
        elif sgn == 1:
            delta = calc - obs
        return delta, sgn

    def getBinPAF (  self, sigma, nu, sgn = 0, allDegeneracies = 1 ) :
        
        # This return the coordinates of the B field
        # in the PAF frame which will give the
        # input nu and sigma values
        #
        # NOTE: There is an eight fold degeneracy,
        #       so up to 8 answers are returned

        #
        # This solves the following equations
        #
        #  x^2 + y^2 + z^2 = 1
        #  sigma = sigma11*x^2 + sigma22*y^2 + sigma33*z^2
        #  nu = nuParalllel/2*(3(sinBeta*x+cosBeta*z)^2 - 1)
        #

        #
        #  Ref:
        # "PISEMA Powder Patterns and PISA Wheels"
        #  Denny, Wang, Cross, & Quine
        #  Journal of Magnetic Resonance 152, 217-226. (2001)
        #

        # compute constants
        pts = []
        epsilons = []
        tanB = math . tan ( self . betaRad )
        secB = 1. / math . cos ( self . betaRad )

        A = self.sigma11 + self.sigma33 * tanB *tanB + \
            self.sigma22 * ( self.sigma33 - self.sigma11 ) / ( self.sigma22 - self.sigma33 )
        C1 = self . sigma22 * ( sigma - self . sigma33 ) / ( self .sigma22 - self .sigma33 )
        
        # There are 4 degeneracies
        #  use epsilon = +1./-1. to enscapsulate

        # check to see sign of dipole is known
        if sgn == -1:
            epsilon1Range = [-1]
        elif sgn == 1:
            epsilon1Range = [1]
        else:
            epsilon1Range = [1, -1]

        for epsilon1 in epsilon1Range :
            rad = (2. * epsilon1 * math.fabs(nu) / self . nuParallel + 1.)/ 3.
            if ( rad >= 0. ) :
                nu0 = tanB * secB * math.sqrt( rad )
                B = self . sigma33 * nu0
                C = self . sigma33 * nu0 * nu0 / ( tanB *tanB ) - sigma + C1
                for epsilon2 in [1, -1 ] :
                    for epsilon3 in [1, -1] :

                        rad = B*B - A*C
                        if rad >= 0. :

                            x = ( epsilon2*B + epsilon3*math.sqrt(rad) ) / A
                            z = epsilon2 * nu0 / tanB - tanB * x
                            if ( ( 1. - x*x - z*z ) >= -self.NumErr ) :
                                # handle numerical error
                                if ((( 1. -x*x - z*z ) >= -self.NumErr ) and \
                                    (( 1. -x*x - z*z ) <= 0. )):
                                    if math . fabs( z ) > math . fabs( x ):
                                        y = 0.
                                        nx = math . sqrt ( 1. - z*z )	
                                        if x < 0.:
                                            x = -nx
                                        else:
                                            x = nx
                                    else:
                                        y = 0.
                                        nz = math . sqrt ( 1. - x*x )	
                                        if z < 0.:
                                            z = -nz
                                        else:
                                            z = nz 
                                else:
                                    y = math . sqrt ( 1. - x*x - z*z )
                                pts . append ( Vector ( x,y,z ) )
                                epsilons . append ( [ epsilon1, epsilon2, epsilon3, 1 ] )
                                if not allDegeneracies:
                                    # just return first valid degeneracy
                                    return pts, epsilons
                                # 4th degeneracy done quickly to save time
                                pts . append ( Vector ( x,-y,z ) )
                                epsilons . append ( [ epsilon1, epsilon2, epsilon3, -1 ] )

        return pts, epsilons

    def projectToFreqPlane ( self, p ) :
        sigma = self . sigma11 * p._x * p._x + self . sigma22 * p._y * p._y  + \
                self . sigma33 * p._z * p._z
        nu = self . nuParallel / 2. * \
             ( 3. * ( \
                 ( math . sin (self.betaRad) * p._x + math . cos ( self.betaRad ) * p._z ) * \
                 ( math . sin (self.betaRad) * p._x + math . cos ( self.betaRad ) * p._z ) ) - 1. )
        return sigma, nu
