#! /usr/bin/env python
#
#  FSUPeptidePlane.py
#
#  A set of classs that allows
#  basic peptide operations:
#   - compuation of phi/psi angles
#   - roatation, translation, etc.
#   - propagation using a given phi/psi angle
#
#
#

import math
import copy

from FSUVector import Vector
from FSUMatrix import Matrix
from FSUPDB import *

class PeptidePlane :

#      Here is what a peptide plane is defined as:
#
#
#             O   -   -   -   -   CA
#              \\                 /
#            /  \\             u3/ \  
#                \\     u2      /   
#            /    C ---------- N   \
#                /              \   
#            /  /u1              \ \ 
#              /                  \
#           PCA   -   -   -   -    H
#
#


    def __init__ ( s ) :
        s . atoms = {
            # canonical peptide plane centered at origin
            'CA' : Vector(            0.0, 0.0,            0.0),
            'N'  : Vector( -1.02530483272, 0.0, -1.02530483272),
            'H'  : Vector( -0.70887143048, 0.0, -1.99918670541),
            'C'  : Vector( -2.32550110593, 0.0, -0.701129492617),
            'O'  : Vector( -2.74960608365, 0.0,  0.464089357158),
            'PCA': Vector( -3.28836130424, 0.0, -1.89016281365)
            } 

    def getU1 ( s, normalize = 0 ) :
        u1 = s . atoms [ 'C' ] . subtract ( s . atoms [ 'PCA' ] )
        if normalize :
            u1 . normalize ( ) 
        return u1

    def getU2 ( s, normalize = 0 ) :
        u2 = s . atoms [ 'N' ] . subtract ( s . atoms [ 'C' ] )
        if normalize :
            u2 . normalize()
        return u2

    def getU3 ( s, normalize = 0 ) :
        u3 = s . atoms [ 'CA' ] . subtract ( s . atoms [ 'N' ] )
        if normalize :
            u3 . normalize ( ) 
        return u3

    def getNH ( s, normalize = 0 ) :
        NH =  s . atoms [ 'H' ] . subtract ( s . atoms [ 'N' ] )
        if normalize:
            NH . normalize ( ) 
        return NH

    def getCO ( s, normalize = 0 ) :
        CO =  s . atoms [ 'O' ] . subtract ( s . atoms [ 'C' ] )
        if normalize :
            CO . normalize ( ) 
        return CO

    def getC2C ( s, normalize = 0 ) :
        C2C = s . atoms [ 'CA' ] . subtract ( s . atoms [ 'PCA' ] )
        if normalize :
            C2C . normalize()
        return C2C

    def getNormal ( s ) :
        # return vector noraml to pp ( u3 x u2 )
        u3 = s . getU3 ( 1 )
        u2 = s . getU2 ( 1 )
        normal = u2 . cross ( u3 )
        return normal

    def calcTorsionAngle ( s, a, b, c, d ) :
        BA = a . subtract ( b )
        BC = c . subtract ( b )
        CD = d . subtract ( c )

        nABC = BC . cross ( BA )
        nBCD = BC . cross ( CD )
        nxn = nABC . cross ( nBCD )
        nABC . normalize ()
        nABC . cleanup ()
        nBCD . normalize ()
        nBCD . cleanup ()

        angle = nABC . dot ( nBCD )
        if math . fabs ( angle ) > 1.0 :
            angle = -1.0
        angle = math . acos ( angle )

        if BC . dot ( nxn ) < 0.0:
            return -angle * 180. / math . pi
        else:
            return angle * 180. / math . pi

    def calcTorAnglesTo ( s, npp ) :
        # calcs phi/psi to npp
        phi = s . calcTorsionAngle ( s . atoms [ 'C' ],
                                     s . atoms [ 'N' ],
                                     s .atoms [ 'CA' ],
                                     npp . atoms [ 'C' ] )
        psi = s . calcTorsionAngle ( s . atoms [ 'N' ], 
                                     s . atoms [ 'CA' ], 
                                     npp . atoms [ 'C' ], 
                                     npp . atoms [ 'N' ] )
        return ( phi, psi )

    def printStats ( s ) :
        # angles
        NCA = s . getU3( 1 )
        NH  = s . getNH( 1 )
        CN  = s . getU2( 1 )
        CO  = s . getCO( 1 )
        PCAC = s . getU1( 1 )
        print "NCA-NH  angle:", math . acos(  NCA . dot( NH  )) * 180./math.pi
        print "CN-NH   angle:", math . acos(   CN . dot( NH  )) * 180./math.pi
        print "CN-NCA  angle:", math . acos(   CN . dot( NCA )) * 180./math.pi
        print "CN-CO   angle:", math . acos(   CN . dot( CO  )) * 180./math.pi
        print "PCAC-CN angle:", math . acos( PCAC . dot( CN  )) * 180./math.pi
        print "PCAC-CO angle:", math . acos( PCAC . dot( CO  )) * 180./math.pi
        
        # distances
        NCA = s . getU3()
        NH  = s . getNH()
        CN  = s . getU2()
        CO  = s . getCO()
        PCAC = s . getU1()
        print "PCA-C distance:", PCAC . length()
        print "C-O   distance:", CO   . length()
        print "C-N   distance:", CN   . length()
        print "N-CA  distance:", NCA  . length()
        print "N-H   distance:", NH   . length()
        
    def translate ( s, dest ) :
        # all motion centered on CA
        offset = dest . subtract ( s . atoms [ 'CA' ] )
        for i in s . atoms . keys() :
            s . atoms [ i ] = s . atoms [ i ] . add ( offset )
            
    def rotate ( s , R ) :
        for i in s . atoms . keys () :
            s . atoms [ i ]  = R . preMultColVec ( s . atoms [ i ] )
            s . atoms [ i ]  . cleanup ( )

    def buildFromResidue ( s, residues, i ) :
      # constructs peptide plane using previous residue
      s . atoms [ 'PCA' ] =  residues [ i - 1 ] . getAlphaCarbon( ) . position
      s . atoms [ 'C'   ] =  residues [ i - 1 ] . getCarbonylCarbon( ) . position
      s . atoms [ 'O'   ] =  residues [ i - 1 ] . getOxygen ( ) . position
      s . atoms [ 'N'   ] =  residues [ i ] . getNitrogen ( ) . position
      s . atoms [ 'H'   ] =  residues [ i ] . getAmineHydrogen ( ) . position
      s . atoms [ 'CA'  ] =  residues [ i ] . getAlphaCarbon( ) . position

    def __repr__( s ):
        sr = "PP\n"
        for i in s . atoms .keys() :
            sr = sr + i + " " + str(s . atoms[i]) + "\n"
        return sr

    def printPlane( s, label = "" ) :
        # printout 4 corner coords
        print label + "{" + str(s.atoms[ 'PCA' ][0]) + ',' + \
              str(s.atoms[ 'PCA' ][1]) + ',' + \
              str(s.atoms[ 'PCA' ][2]) + '},'
        print label + "{" + str(s.atoms[ 'O' ][0]) + ',' + \
              str(s.atoms[ 'O' ][1]) + ',' + \
              str(s.atoms[ 'O' ][2]) + '},'
        print label + "{" + str(s.atoms[ 'CA' ][0]) + ',' + \
              str(s.atoms[ 'CA' ][1]) + ',' + \
              str(s.atoms[ 'CA' ][2]) + '},'
        print label + "{" + str(s.atoms[ 'H' ][0]) + ',' + \
              str(s.atoms[ 'H' ][1]) + ',' + \
              str(s.atoms[ 'H' ][2]) + '},'          
       
class Propagator:

    # This class is the peptide plane Propagator
    # tranforms one peptide plane to the
    # next ( or previous ) given a
    # phi and psi angle
    # - contains lots of support code
    #

    def __init__ ( s, nmr ) :
        s . phi = 0.
        s . psi = 0.
        s . nmr = nmr
        s . X = Vector ( 1., 0., 0. )
        s . Y = Vector ( 0., 1., 0. )
        s . Z = Vector ( 0., 0., 1. )
        s . Origin = Vector ( 0., 0., 0. )
        s . extAngleAlpha = 65.
        s . extAngleBeta = 59.
        s . extAngleGamma = 70.
        s . P = Matrix ( ) 
        s . RR = Matrix ( )
        s . buildRRMatrix () 
	s . NUM_ERR = 0.0001

    def buildPropagatorMatrix ( s, phi, psi ):
        #
        #  P = Rx(phi) * Rz (gamma) * Rx ( psi ) * Rz (alpha) * Rx ( 180 ) * Rz (beta)
        #  where alpha, beta, and gamma are defined in Denny paper
        #       
        R = Matrix()
        s . P . setRotX ( phi * math . pi / 180. )
        R . setRotZ ( s . extAngleGamma * math . pi / 180. )
        s . P = s . P . mult ( R )
        R . setRotX ( psi * math . pi / 180. )
        s . P = s . P . mult ( R )
        s . P = s . P . mult ( s . RR )
        
    def buildRRMatrix ( s ) :
        alpha = 65.
        beta = 59.
        omega = 180.
        s . RR . setRotZ ( s . extAngleAlpha * math . pi / 180. )
        R = Matrix ( ) 
        R . setRotX ( omega * math . pi / 180. )
        s . RR = s . RR . mult ( R )
        R . setRotZ ( s . extAngleBeta * math . pi / 180. )
        s . RR = s . RR . mult ( R )
        
    def translateResidue ( s, residue, dest ) :
        # translate to dest
        CA = residue . getAlphaCarbon ( ) . position
        offset = dest . subtract ( CA ) 
        for atom in residue :
            atom . position = atom . position . add ( offset )

    def alignResidueWithVectors ( s, residue, O, A, B ):

        # align a residue with
        #    CA-N bond pointing in the A direction
        #    CA-C bond in the A-B plane
        #    CA-N X CA-C direction = A X B direction
        #    translate CA to O
        N = residue . getNitrogen ( ) . position
        C = residue . getCarbonylCarbon ( ) . position
        CA = residue . getAlphaCarbon ( ) . position 
        a = N . subtract ( CA )
        b = C . subtract ( CA )
        a . normalize ()
        b . normalize ()
        a . cleanup ()
        b . cleanup () 

        # move to origin, rotate, and then to O
        s . translateResidue ( residue, s . Origin )
        R = s . alignVectorSets ( A, B, a, b )
        for atom in residue :
            atom . position = R . preMultColVec ( atom . position )
            atom . position . cleanup ( )
        s . translateResidue ( residue , O )
            
    def alignPeptidePlaneWithVectors ( s, pp, O, A, B ):

        # align a peptide plane:
        #    U3(N-CA) pointing in the B direction
        #    N-H bond in the A-B plane
        #    U3 X N-H direction = A X B direction
        #    translate CA to O
        a = pp . getNH ( )
        b = pp . getU3 ( )
        a . normalize ()
        a . cleanup ()
        b . normalize()
        b . cleanup ( ) 

        # translate N to origin, rotate and then translate to O
        pp . translate ( b )
        R = s . alignVectorSets ( A, B, a, b )
        pp . rotate ( R )

        pp . translate ( O )

    def alignVectorSets ( s, A, B, a, b ):
                
      # computes matrix to align vector set
      # { a, b } with { A, B }
      # such that
      #  a pointing in A direction
      #  b point in B direction
      #  axb = AxB

      alpha = math . acos ( a . dot ( A ) )
      axis = a . cross ( A )
      axis . normalize ( )
      axis . cleanup ( )
      R = Matrix ()
      R . setRotVector ( alpha, axis )
      
      b = R . preMultColVec ( b )

      # CA - C bond to be in AB plane
      ANormal = A . cross (  B )
      ANormal . normalize () 
      ANormal . cleanup ()
        
      CNormal = A . cross ( b ) 
      CNormal . normalize () 
      CNormal . cleanup ()

      x = ANormal . dot ( CNormal )
      y = math . sqrt ( 1.0 - x*x )
      PP = Vector ( CNormal . _x, CNormal . _y, CNormal . _z )
      PP . scale ( x )
      QQ = ANormal . subtract ( PP )
      QP = PP . cross ( QQ )
        
      if A . dot ( QP ) < 0.0:
          if x > 0.0 :
              y = -y
      else:
          if x < 0.0 :
              y = -y
            
      theta = math . atan2 ( y, x )     
      R2 = Matrix () 
      R2 . setRotVector ( theta, A )

      return ( R2 . mult ( R ) ) # composite rotation matrix

    def alignResidueWithResidue ( s, a, b, translate = 0 ):

        # just align CA - N bond to be co-incident 
        N = b . getNitrogen ( ) . position
        C = b . getCarbonylCarbon ( ) . position
        CA = b . getAlphaCarbon ( ) . position 

        if translate :
            O = CA
        else:
            O = a . getAlphaCarbon ( ) . position  
        A = N . subtract ( CA )
        B = C . subtract ( CA ) 
        A . normalize ( )
        A . cleanup ( )
        B . normalize ( )
        B . cleanup ( ) 

        s . alignResidueWithVectors ( a, O, A, B )

    def getAlignmentTransform ( s, A, B ) :
        # return Rotation Matrix which transforms A to B
        a = A
        b = B
        a . normalize()
        b . normalize()
        a . cleanup()
        b . cleanup()
        axis = a . cross ( b )
        if ( math . fabs ( axis . length ( ) ) <= s . NUM_ERR ) :
            if a . dot ( b ) > 0. :
                R = Matrix ( )
                return R
            else:
                R = Matrix ()
                R . setRotZ ( math . pi )
                return R
        axis . normalize()
        axis . cleanup ()
        angle = math . acos ( a . dot ( b ) )
        if angle < 0.:
            angle = math . pi + angle
        R = Matrix ( )
        R . setRotVector ( angle, axis )
        return R

    def alignWithPAF ( s, pp, BinPAF, BinLab, betaRad ) :
        # this is found by comparing BinPAF with BinLab
        # and finding transform to make them coincident
        # NOTE : BinLab assumed to be ( 0, 0, 1 )
        R = s . getAlignmentTransform ( BinPAF, BinLab )
        # MF in local frame is Identity matrix
        MF = Matrix()
        MF = R . mult ( MF )
        sigma11 = MF . getCol ( 0 ) 
        sigma22 = MF . getCol ( 1 )
        sigma33 = MF . getCol ( 2 ) 
        # now have MF in Lab Frame
        # now rotate pp to have u1xu2 coinicident with sigma22
        u1 = pp . getNH ( 1 )
        u2 = pp . getU3 ( 1 )
        b = u1 . cross ( u2 )
        b . normalize()
        b . cleanup()
        R = s . getAlignmentTransform ( b, sigma22 )
        # rotate around N atom
        offset = pp . getU3 ( ) # unnormalized!!
        pp . translate ( offset )
        pp . rotate ( R )

        MF = s . nmr . computePAF ( pp . atoms [ 'N' ],
                                    pp . atoms [ 'H' ],
                                    pp . atoms [ 'CA' ],
                                    0., 0. )

        nsigma11 = MF . getCol ( 1 )
        nsigma22 = MF . getCol ( 2 )
        nsigma33 = MF . getCol ( 0 ) 

        # now U1 is in sigma11 sigma33 plane
        # offset by beta
        u1 = pp . getNH ( 1 )
        u2 = pp . getU3 ( 1 )
        R . setRotVector ( betaRad, sigma22 )
        betau1 = R . preMultColVec ( sigma33 )
        R = s . getAlignmentTransform ( u1, betau1 ) 
        pp . rotate ( R )
        offset . scale ( -1. )
        pp . translate ( offset )

        if 0:
            MF = s . nmr . computePAF ( pp . atoms [ 'N' ],
                                        pp . atoms [ 'H' ],
                                        pp . atoms [ 'CA' ],
                                        s .nmr . toRad ( 17.) , 0. )
            
            nsigma11 = MF . getCol ( 1 )
            nsigma22 = MF . getCol ( 2 )
            nsigma33 = MF . getCol ( 0 ) 
            u1 = pp.getU2 ( 1 )
            print "u1", \
                  180.* math . acos ( u1 . dot ( nsigma11 ) ) / math.pi, \
                  180.* math . acos ( u1 . dot ( nsigma22 ) ) / math.pi, \
                  180.* math . acos ( u1 . dot ( nsigma33 ) ) / math.pi
            u2 = pp.getU3 ( 1 )
            print "u2", \
                  180.* math . acos ( u2 . dot ( nsigma11 ) ) / math.pi, \
                  180.* math . acos ( u2 . dot ( nsigma22 ) ) / math.pi, \
                  180.* math . acos ( u2 . dot ( nsigma33 ) ) / math.pi
            nh = pp.getNH ( 1 )
            print "nh", \
                  180.* math . acos ( nh . dot ( nsigma11 ) ) / math.pi, \
                  180.* math . acos ( nh . dot ( nsigma22 ) ) / math.pi, \
                  180.* math . acos ( nh . dot ( nsigma33 ) ) / math.pi
            print "cn-nh", \
                  180.* math . acos ( u1 . dot ( nh ) ) / math.pi


    def printDistances ( s, residues, i ) :
        u1, u2, u3 = s . computeUVectors ( residues, i ) 
        print u1.length( ) , u2.length ( ), u3.length()

    def getInternalAngle ( s, A, B ) :
        A . normalize  ( )
        A . cleanup ( )
        B . normalize ( )
        B . cleanup ( ) 
        angleRad =  math . acos ( A . dot ( B ) )
        return ( 180. * angleRad / math . pi  )

    def computeExternalAngles ( s, residues, rIndex ) :
        N = residues [ rIndex ] . getNitrogen ( ) . position
        PC = residues [ rIndex - 1 ]  . getCarbonylCarbon ( ) . position
        CA = residues [ rIndex ]  . getAlphaCarbon ( ) . position 
        PCA = residues [ rIndex - 1 ]  . getAlphaCarbon ( ) . position 
        C = residues [ rIndex ]  . getCarbonylCarbon ( ) . position

        # alpha
        A = PC . subtract ( PCA )
        B = N . subtract ( PC )
        alpha = s . getInternalAngle ( A, B ) 
        # beta
        A = N . subtract ( PC )
        B = CA . subtract ( N )
        beta = s . getInternalAngle ( A, B ) 
        # gamma
        A = CA . subtract ( N )
        B = C . subtract ( CA )
        gamma = s . getInternalAngle ( A, B ) 

        return alpha, beta, gamma

    def propagatePeptidePlane ( s, pp, phi, psi, backwards = 0 ) :
        # transform is of the form
        #
        #   PPnew = [ MF-1 ] [ Propagator Matrix ] [ MF ]  PPold
        #
	#  NOTE: just using NMR's computePAF with alpha, beta = 0.
	#        to get Molecular Frame (MF)
        MF = s . nmr . computePAF ( pp . atoms [ 'N' ],
                                    pp . atoms [ 'CA' ],
                                    pp . atoms [ 'C' ],
                                    0., 0. ) 

        MF . transpose ( )
        s . buildPropagatorMatrix ( phi, psi )

        if backwards:
            s . P . transpose ( ) # take inverse to propagate backwards
        
        R = s . P . mult ( MF ) 
        MF . transpose ( )
        R = MF . mult ( R ) 
        ppnew = copy.deepcopy ( pp )
        ppnew . rotate ( R )

        if backwards:
            # PULL new peptide plane 
            #    translate ppnew's CA to pp PCA
            #    dest = ppPCA + ( ppPCA - ppCA )
            dest = pp . atoms [ 'CA' ] . add (
                pp . atoms [ 'PCA' ] . subtract (  pp . atoms [ 'CA' ] ) )
        else:
            # push new peptide plane 
            #    translate ppnew's PCA to pp CA
            #    dest = ppCA + ( ppnewCA - ppnewPCA )
            dest = pp . atoms [ 'CA' ] . add (
                ppnew . atoms [ 'CA' ] . subtract (  ppnew . atoms [ 'PCA' ] ) )

        ppnew . translate ( dest )
        return ppnew   
    
    def Tors( s, v1v2, v1v3, v2v3 ) :
        # all vectors are assumed normalized
        # see Ref below for explanation
        ts = []
        x = -v1v3 + v1v2*v2v3
        y = 1. - v1v2*v1v2 - v2v3*v2v3 - v1v3*v1v3 + 2*v1v2*v2v3*v1v3
        if y > 0. :
            y = math . sqrt ( y )
            x = x / math . sqrt ( x*x + y*y )
            ts . append ( math . acos ( x ) * 180. / math . pi )
            ts . append ( -ts[-1] )
        return ts[:]

    def toRad( s, deg ) :
        return deg * math.pi/180.


    def getBFieldDotProducts ( s,  BinPAF1, BinPAF2, betaRad ) :
        # this is the geometric portion of the
        # torsion angle calculations between 2 NMR
        # resonances

        u1B = math . cos ( -betaRad + s . toRad( 32.  )) * BinPAF1[0] + \
              math . cos (  betaRad + s . toRad( 58.  )) * BinPAF1[2]
        u2B = math . cos (  betaRad + s . toRad( 27.  )) * BinPAF1[0] + \
              math . cos (  betaRad + s . toRad( 117. )) * BinPAF1[2]

        u3B = math . cos (  betaRad + s . toRad( 33.  )) * BinPAF2[0] + \
              math . cos (  betaRad + s . toRad( 123. )) * BinPAF2[2]
        u4B = math . cos ( -betaRad + s . toRad( 32.  )) * BinPAF2[0] + \
              math . cos (  betaRad + s . toRad( 58.  )) * BinPAF2[2]

        return  u1B, u2B, u3B, u4B

    def gramian ( s, x, y, z ) :
        a = 1. - x*x - y*y - z*z + 2*x*y*z
        if a >= 0. :
            return ( math . sqrt ( a ) )
        elif a >= -0.01:
            return 0.
        else:
            return a


    def arg (s,  x,y ) :
        x = x / math . sqrt ( x*x + y*y )
        angle = math . acos ( x ) * 180. / math . pi
        if y < 0.:
            angle = -1 * angle
        return angle

    def computeTorsionsByChiralities ( s, BinPAF1, BinPAF2, betaRad, printOut = 0, carbonChiralities = 0 ) :

        # build directional unit vertors 
        pp = PeptidePlane ( )
        u1 = pp . getU2 ( 1 )
        u2 = pp . getU3 ( 1 )
        u3 = pp . getU1 ( 1 )
        u4 = pp . getU2 ( 1 )
        
        v1v2 = u1 . dot( u2 )
        v2v3 = math . cos( s . toRad( 70. ))
        v3v4 = u3 . dot( u4 )        

        mu1, mu2, mu3, mu4 = s . getBFieldDotProducts( BinPAF1, BinPAF2, betaRad )

        psi1s = []
        psi2s = []
        phi1s = []
        phi2s = []
        tors = []
        exact = []
        grams = []
        eps = []

        # there are 2^5 = 32 degeneracies
        # use epsilons to keep track of them
        if carbonChiralities:
            carbonChiral = [1, -1]
        else:
            carbonChiral = [1]

        for epsilon_t1 in [-1,1] :
            for epsilon_t1p in [-1,1] :
                for epsilon_tc in carbonChiral :
                    for epsilon_t2 in [-1,1] :
                        for epsilon_t2p in [-1,1] :

                            inexact = 0
                            
                            gram1 = s . gramian( mu1, mu2, v1v2 )
                            if gram1 >= 0.:
                                theta1 = s . arg( -epsilon_t1*mu1 + epsilon_t1*mu2*v1v2, \
                                                  epsilon_t2*gram1)
                            else:
                                inexact = 1
                                theta1 = 0.

                            gram2 = s . gramian( epsilon_t1*mu2, epsilon_t1p*mu3, v2v3 )
                            if gram2 >= 0.:
                                theta2 = s . arg( epsilon_t1p*mu3 - epsilon_t1*mu2*v2v3, \
                                                  -epsilon_tc*gram2 )
                            else:
                                inexact = 1
                                theta2 = 0.

                            gram3 = s . gramian( epsilon_t1*mu2, epsilon_t1p*mu3, v2v3 )
                            if gram3 >= 0.:
                                theta3 = s. arg( -epsilon_t1*mu2 + epsilon_t1p*mu3*v2v3, \
                                                 epsilon_tc*gram3 )
                            else:
                                inexact = 1
                                theta3 = 0.


                            gram4 = s . gramian( mu3, mu4, v3v4 )
                            if gram4 >= 0.:
                                theta4 = s . arg( epsilon_t1p*mu4 - epsilon_t1p*mu3*v3v4, \
                                                  -epsilon_t2p*gram4 )
                            else:
                                inexact = 1
                                theta4 = 0.

                            psi = theta3 + theta4
                            if ( psi > 180. ) :
                                psi = psi - 360.
                            elif ( psi < -180.) :
                                psi = psi + 360.

                            phi = theta1 + theta2
                            if ( phi > 180. ) :
                                phi = phi - 360.
                            elif ( phi < -180.) :
                                phi = phi + 360.
                                
                            if printOut:
                                print "%10.2f & %10.2f & %10.2f & %10.2f & %10.2f" \
                                      " & %10.2f & %10.2f & %10.2f & %10.2f & %10.2f \\\\" % \
                                      ( phi, psi, \
                               #         epsilon_t1, epsilon_t1p, epsilon_tc, 
                               #         epsilon_t2, epsilon_t2p, 
                                        theta1, theta2, theta3, theta4, \
                                        gram1, gram2, gram3, gram4 )
                            tors . append ( (phi,psi ) )
                            exact . append ( not inexact )
                            grams . append ( gram2 )
                            eps . append ( (epsilon_t1, epsilon_t1p, epsilon_tc, epsilon_t2, epsilon_t2p) )
                                           
        return ( tors, exact, grams, eps )

    def computeTorsionsByNMR( s, BinPAF1, BinPAF2, betaRad ) :
        # Computes torsion angles
        # in between 2 peptide planes
        # with B fields in each.
        # Due to the nature of degneracies
        # 4 possible phis and phis are returned,
        # 16 combinations of (phi,psi)
        #
        # Ref. "Transmembrane Domain of M2 Protein...",
        # Song et al, Biophysical Journal, 79, Aug 2000, pp.767-775.

        # build directional unit vertors 
        pp = PeptidePlane ( )
        u1 = pp . getU2 ( 1 )
        u2 = pp . getU3 ( 1 )
        u3 = pp . getU1 ( 1 )
        u4 = pp . getU2 ( 1 )

        u1u2 = u1 . dot( u2 )
        u2u3 = math . cos( s . toRad( 70. ))
        u3u4 = u3 . dot( u4 )        
 
        u1B, u2B, u3B, u4B = s . getBFieldDotProducts( BinPAF1, BinPAF2, betaRad )

        # phi
        phis1 = s . Tors(  u1u2,  u1B, u2B )
        phis2 = s . Tors(  -u2B, -u3B, u2u3 )
        phis = []
        indeterminatePhi = 0
        phithetas = []
        psithetas = []
        if len( phis1 ) :
            for ip in range( len( phis1 )):
                if len( phis2 ):
                    for ipp in range( len( phis2 )):
                        angle = phis1[ip] + phis2[ipp]
                        if ( angle > 180. ) :
                            angle = angle - 360.
                        elif ( angle < -180.) :
                            angle = angle + 360.
                        phis . append( angle )
                        phithetas . append ( (phis1[ip], phis2[ipp] ))
                else:
                    indeterminatePhi = 1
                    phis . append( phis1[ip] )
                    phithetas . append ( (phis1[ip], 500.00 ))
        else:
            indeterminatePhi = 1
            if len( phis2 ):
                for ipp in range( len( phis2 )):
                    phis . append( phis2[ipp] )
                    phithetas . append ( ( 500.00, phis2[ipp] ))

        # psi
        psis1 = s . Tors( u2u3,  u2B,  u3B )
        psis2 = s . Tors( -u3B, -u4B, u3u4 )
        psis = []
        indeterminatePsi = 0
        if len( psis1 ):
            for ip in range( len( psis1 )):
                if len( psis2 ):
                    for ipp in range( len( psis2 )):
                        angle = psis1[ip] + psis2[ipp]
                        if ( angle > 180. ) :
                            angle = angle - 360.
                        elif ( angle < -180.) :
                            angle = angle + 360.
                        psis . append( angle )
                        psithetas . append ( (psis1[ip], psis2[ipp] ))
                else:
                    indeterminatePsi = 1
                    psis . append( psis1[ ip ] )
                    psithetas . append ( (psis1[ip], 500.00 ))
        else:
            indeterminatePsi = 1
            if len( psis2 ):
                for ipp in range( len( psis2 )):
                    psis . append( psis2[ipp] )
                    psithetas . append ( ( 500.00, psis2[ipp] ))

        return ( phis, psis, indeterminatePhi, indeterminatePsi, phithetas, psithetas )
        
        
