#! /usr/bin/env python
#
# computes ssNMR synthetic data
#

import sys
import math
import getopt
import string
import re
from random import Random 

from FSUVector import Vector
from FSUMatrix import Matrix
from FSUPDB import *

class ssNMR :

#
# Computes ssNMR PISEMA data
#
#  Main reference is:
#     "Atomic Refinement Using Orientational Restraints from ssNMR"
#     Bertram, Quine, Chapman and Cross
#     Journal of Magnetic Resonance, 147, 9-16. (2000)
#
#
    def __init__ ( self ) :
        self . reflect = 0
        self . sigmas = []
        self . nus = []
        self . numbers = []
        self . types = []
        self . rbetas = []
        self . dipangles = []
        self . sigma11 = 0.
        self . sigma22 = 0.
        self . sigma33 = 0.
        self . nuParallel = 0.
        self . betaRad = 0.
        self . alphaRad = 0.
        # default from paper
        self . CnuParallel = 1.271 
        self . NHnuParallel = 11.335
        self . IndolenuParallel = 10.17 
        self . CbetaRad = self . toRad ( 35. ) 
        self . NHbetaRad = self . toRad ( 17. )
        self . IndolebetaRad = self . toRad ( 25. ) 
        self . NHsigma11 = 30.  
        self . NHsigma22 = 60. 
        self . NHsigma33 = 205.
        self . Csigma11 = 90.  
        self . Csigma22 = 175. 
        self . Csigma33 = 245.
        self . Indolesigma11 = 45.
        self . Indolesigma22 = 115.
        self . Indolesigma33 = 165.
        self . nc = 0
        self . nh = 0
        self . indole = 0
        self . rawbeta = 0
        self . PAF = 0
        self . dipolarAngle = 0
        self . Bs = []
        self . thetas = []
        # correlation flags
        self . correlateNone = 0
        self . correlate1D = 1
        self . correlate2D = 2
	self . NumErr = 0.000001

    def setC ( self ) :
        self . nuParallel = self . CnuParallel
        self . betaRad = self . CbetaRad
        self . sigma11 = self . Csigma11
        self . sigma22 = self . Csigma22
        self . sigma33 = self . Csigma33
        self . nc = 1
        self . nh = 0
        self . indole = 0

    def setNH ( self ) :
        self . nuParallel = self . NHnuParallel
        self . betaRad = self . NHbetaRad
        self . sigma11 = self . NHsigma11
        self . sigma22 = self . NHsigma22
        self . sigma33 = self . NHsigma33  
        self . nh = 1
        self . nc = 0 
        self . indole = 0

    def setIndole ( self ):
        self . nuParallel = self . IndolenuParallel
        self . betaRad = self . IndolebetaRad
        self . sigma11 = self . Indolesigma11
        self . sigma22 = self . Indolesigma22
        self . sigma33 = self . Indolesigma33  
        self . nh = 0
        self . nc = 0
        self . indole = 1

    def toRad ( self, x ) :
        return ( x * math.pi / 180. ) 

    def toDegree ( self, x ) :
        return ( x * 180. / math.pi )

    def error ( self, a, b ) :
        return ( ( float ( a ) - float ( b ) ) / float ( b )  * 100. )

    def setReflection ( self, reflect ) :
        self . reflect = reflect

    def setBetaDetermination ( self, bb ) :
        self . rawbeta = bb

    def setNHParameters ( self, betaRad, nu, sigma11, sigma22, sigma33 ) :
        self . NHnuParallel = nu
        self . NHbetaRad = betaRad
        self . NHsigma11 = sigma11
        self . NHsigma22 = sigma22
        self . NHsigma33 = sigma33

    def setCParameters ( self, betaRad, nu, sigma11, sigma22, sigma33 ) :
        self . CnuParallel = nu
        self . CbetaRad = betaRad
        self . Csigma11 = sigma11
        self . Csigma22 = sigma22
        self . Csigma33 = sigma33
        
    def setIndoleParameters ( self, betaRad, nu, sigma11, sigma22, sigma33 ) :
        self . IndolenuParallel = nu
        self . IndolebetaRad = betaRad
        self . Indolesigma11 = sigma11
        self . Indolesigma22 = sigma22
        self . Indolesigma33 = sigma33
    
    def getNu ( self, sigma, neg = 0 ) :
	sigmaBar = ( sigma - self . sigma11 ) / ( self . sigma33 - self . sigma11 ) 
        if (sigmaBar < 0. and sigmaBar > -self.NumErr) :
            sigmaBar = 0.
	nuBar = math . cos ( 2. * self . betaRad ) * sigmaBar + \
		math . sin ( self . betaRad ) * math . sin ( self . betaRad ) 
	if neg :
	   nuBar = nuBar - math . sqrt ( sigmaBar * ( 1.0 - sigmaBar ) * \
			math . sin ( 2. * self . betaRad ) *  \
			math . sin ( 2. * self . betaRad ) )
	else :
	   nuBar = nuBar + math . sqrt ( sigmaBar * ( 1.0 - sigmaBar ) * \
			math . sin ( 2. * self . betaRad ) * \
			math . sin ( 2. * self . betaRad ) )
	# perform inverse transform
	if ( nuBar > 1. ) :
		print "NuBar Error ", sigmaBar, nuBar1, nuBar
	nu = self . nuParallel / 2. * ( 3. * nuBar - 1. ) 


	return nu

    def getDipolePowder ( self, s, rezs, B ) :
        if self . nc :
            self . getCDipolePowder ( s, rezs, B )
        if self . nh :
            self . getNHDipolePowder ( s, rezs, B )
        if self . indole :
            self . getIndoleDipolePowder ( s, rezs, B )             

    def getCDipolePowder ( self, s, rezs, B ) :
        for r in rezs :
            ii = string. rfind ( r, ":" )
            f = int ( r [ :ii ] ) - 1
            l = int ( r [ ( ii + 1 ): ] ) 
            
            if ( f == -1 ) :
                f = 0
                
            if ( l == 0 ) :
                l = len ( s . residues )

            for i in range ( f, l ) :
	  
                if ( hasattr( s . residues [ i ] , 'is_amino_acid') and \
                     hasattr( s . residues [ i - 1 ] , 'is_amino_acid') and \
                     # N is bonded differently in Proline
                     s . residues [ i ]  . name != "PRO" ) :
                    
                    N = s . residues [ i ] . getNitrogen ( ) . position
                    C1 = s . residues [ i - 1 ] . getCarbonylCarbon ( ) . position
                    CA = s . residues [ i ] . getAlphaCarbon ( ) . position 
                    
                    theta, enu = self . calculateNuAndTheta ( B, N, C1 )
                    pas = self . computePAF ( N, C1, CA, self . betaRad, self . alphaRad )
                    esigma = self . computeChemShift ( pas, B )
                    
                    self . sigmas . append ( esigma )
                    self . nus .append ( enu ) 
                    self . numbers . append ( s . residues [ i ] . number )
                    self . types . append( self . convert( s . residues [ i] . name) )
                    
                    if self . dipolarAngle :
                        self . dipangles . append ( theta );

    def getNHDipolePowder ( self, s, rezs, B ) :
        for r in rezs :
            ii = string. rfind ( r, ":" )
            f = int ( r [ :ii ] ) - 1
            l = int ( r [ ( ii + 1 ): ] ) 
            
            if ( f == -1 ) :
                f = 0
            if ( l == 0 ) :
                l = len ( s . residues )
                
            for i in range ( f, l ) :

                if ( hasattr( s . residues [ i ] , 'is_amino_acid' ) and \
                     hasattr( s . residues [ i - 1 ] , 'is_amino_acid' ) and \
                     # no H in Proline
                     s . residues [ i ]  . name != "PRO" ) :
                    
                    Hatom = s . residues [ i ] . getHydrogen ( )
                    HNatom = s . residues [ i ] . getAmineHydrogen ( ) 
                    Natom = s . residues [ i ] . getNitrogen ( ) 
                    CAatom = s . residues [ i ] . getAlphaCarbon ( )
                    
                    if Natom . isNull ( ) :
                        print "Error - could not find Nitrogen in residue " + str(i)
                        sys .exit ( 0 )
                    else:
                        N = Natom . position
                        
                    if Hatom . isNull ( ) and HNatom . isNull ( ) :
                        print "Error - could not find Hydrogen in residue " + str(i)
                        sys . exit ( 0 )
                    else:
                        if not HNatom . isNull () :
                            H = HNatom . position  # use this one first   
                        else:
                            H = Hatom . position

                    if CAatom . isNull ( ) :
                        print "Error - could not find Alpha Carbon in residue " + str(i)
                        sys . exit ( 0 )
                    else:
                        CA = CAatom . position
            
                    theta, enu = self .calculateNuAndTheta ( B, N, H )
                    if self . rawbeta:
                        if i == 0:
                            print "skipping first residue - no previous CO"
                            continue
                        C1atom = s . residues [ i - 1 ] . getCarbonylCarbon ( ) 
                        C1 = C1atom . position
                        # compute exact beta angle
                        #  - assume Beta is 105 for NC 
                        #    thus Beta for NH is N-C-H angle - 105
                        NH = H . subtract ( N )
                        NH . normalize ( ) 
                        NC = C1 . subtract ( N )
                        NC . normalize ( )
                        cosAngle = NC . dot ( NH )
                        rawBeta = math . acos ( cosAngle ) - self . toRad ( 105. )
                        self . rbetas . append ( rawBeta )
                        pas = self .computePAF ( N, H, CA, rawBeta, self . alphaRad )
                    else:
                        pas = self .computePAF ( N, H, CA, self . betaRad, self . alphaRad )
                    esigma = self .computeChemShift ( pas, B )

                    self . sigmas . append ( esigma )
                    self . nus . append ( enu )
                    self . numbers . append ( s . residues [ i ] . number )
                    self . types . append ( self . convert( s . residues[ i] . name ) )
                    
                    if self . dipolarAngle :
                        self . dipangles . append ( theta );

    def getIndoleDipolePowder ( self, s, rezs, B ):

        for r in rezs :
            ii = string. rfind ( r, ":" )
            f = int ( r [ :ii ] ) - 1
            l = int ( r [ ( ii + 1 ): ] ) 
            
            if ( f == -1 ) :
                f = 0
                
            if ( l == 0 ) :
                l = len ( s . residues )
                
            for i in range ( f, l ) :
	  
                if ( hasattr( s . residues [ i ] , 'is_amino_acid') and \
                     hasattr( s . residues [ i - 1 ] , 'is_amino_acid') and \
                     ( s . residues [ i ] . name == "HIS" or \
                       s . residues [ i ] . name == "TRP" )):

                    ia = 0
                    ib = 0
                    ic = 0
                    if s . residues [ i ] . name == "HIS" :
                        for a in s . residues[i] . atom_list:
                            if a . name == 'HE2':
                                BB = a . position
                                ib = 1
                            if a . name == 'CE1':
                                CC = a . position
                                ic = 1
                            if a . name == 'NE2':
                                AA = a . position
                                ia = 1
                                
                    if s . residues [ i ] . name == "TRP" :
                        for a in s . residues[i] . atom_list:
                            if a . name == 'HE1':
                                BB = a . position
                                ib = 1
                            if a . name == 'CE2':
                                CC = a . position
                                ic = 1
                            if a . name == 'NE1':
                                AA = a . position
                                ia = 1

                    if ia and ib and ic :

                        theta, enu = self . calculateNuAndTheta ( B, AA, BB )
                        pas = self . computePAF ( AA, BB, CC, self . betaRad, self . alphaRad )
                        esigma = self . computeChemShift ( pas, B )
                        
                        self . sigmas . append ( esigma )
                        self . nus .append ( enu ) 
                        self . numbers . append ( s . residues [ i ] . number )
                        
                        if self . dipolarAngle :
                            self . dipangles . append ( theta );
                    else:
                        print "Error - could not find HE1, CE2 in residue " + str(i)
                        sys . exit ( 0 )

    def calculateNuAndTheta ( self, B, a, b ) :
        ab = b . subtract ( a )
	ab . normalize ( )
        theta = B . dot ( ab ) 
        costhetasq = theta * theta 
	nu = self . nuParallel / 2. * ( 3. * costhetasq - 1. )
        theta = math . acos ( theta )
	return ( theta, nu ) 
   
    def computePAF ( self, A, B, C, betaRadians, alphaRadians ) :
        # see Denny pp. 11-12
	f = self . computeF ( A, B, C )
	mbeta = Matrix( ) 
	mbeta . setRotZ( -betaRadians )
	nalpha = Matrix( ) 
	nalpha . setRotX( -alphaRadians )
	fm = f . mult ( mbeta )   
	pas = fm . mult ( nalpha )
	return pas

    def computeF ( self, A, B, C ) :
        u1 = B . subtract ( A )
	u1 . normalize ( ) 
	u2 = C . subtract ( A )
	u2 . normalize ( )
	b = u1 . cross ( u2 )
	b . normalize ()
	n = b . cross ( u1 )
	n . normalize ( )
	F = Matrix ( ) 
	F . setCol ( 0, u1 ) 
	F . setCol ( 1, n )
	F . setCol ( 2, b )
        return F

    def computeChemShift ( self, pas, B, index11 = 1, index22 = 2, index33 = 0 ) :
	sigma11v = pas . getCol ( index11 ) 
	sigma22v=  pas . getCol ( index22 ) 
	sigma33v = pas . getCol ( index33 )
        if self . PAF :
            print "SIGMA 11: ", sigma11v
            print "SIGMA 22: ", sigma22v
            print "SIGMA 33: ", sigma33v
            Bpas = pas . postMultRowVec ( B ) 
            print "B in PAF", Bpas [index11], Bpas[index22], Bpas[index33]
	B11dot = sigma11v . dot ( B ) 
	B22dot = sigma22v . dot ( B ) 
	B33dot = sigma33v . dot ( B )
	return ( self . sigma11 * B11dot * B11dot + \
		 self . sigma22 * B22dot * B22dot + \
		 self . sigma33 * B33dot * B33dot )

    def getTrueBinPAF ( self, pas, B, index11 = 1, index22 = 2, index33 = 0  ) :
         Bpas = pas . postMultRowVec ( B )
         trueB = Vector ( Bpas[index11], Bpas[index22], Bpas[index33] )
         return trueB

    def determineDipoleSign ( s, nu, sigma, correlation ) :
        if correlation == s . correlateNone:
            return 0
        elif correlation == s . correlate1D:
            if nu >= s .nuParallel / 2. :
                return 1
            else:
                return 0
        elif correlation == s . correlate2D:
            # 2D correlation
            sigmaBar = ( sigma - s . sigma11 ) / ( s . sigma33 - s . sigma11 )
            if (sigmaBar < 0. and sigmaBar > -0.0001 ) :
                sigmaBar = 0.
            nuPos = s . getNu ( sigma )
            nuNeg = s . getNu ( sigma, -1 )
            ellipse = 0
            if nu >= nuNeg and nu <= nuPos:
                ellipse = ellipse + 1
            if nu >= -nuPos and nu <= -nuNeg:
                ellipse = ellipse - 1
            return ellipse
            
    def getCorrelatedError ( s, calc, obs, sigma, correlate ) :
        # this return the correlated dipole
        # error based on looking at the corresponding cs
        # values and determining where on the freq plane
        # the data point
        sgn = s . determineDipoleSign ( obs , sigma, correlate )
        if sgn == 0:
            delta = math . fabs ( calc ) - obs
        elif sgn == -1:
            delta = calc + obs
        elif sgn == 1:
            delta = calc - obs
        return delta, sgn

    def getBinPAF (  self, sigma, nu, sgn = 0, allDegeneracies = 1 ) :
        
        # This return the coordinates of the B field
        # in the PAF frame which will give the
        # input nu and sigma values
        #
        # NOTE: There is an eight fold degeneracy,
        #       so up to 8 answers are returned

        #
        # This solves the following equations
        #
        #  x^2 + y^2 + z^2 = 1
        #  sigma = sigma11*x^2 + sigma22*y^2 + sigma33*z^2
        #  nu = nuParalllel/2*(3(sinBeta*x+cosBeta*z)^2 - 1)
        #

        #
        #  Ref:
        # "PISEMA Powder Patterns and PISA Wheels"
        #  Denny, Wang, Cross, & Quine
        #  Journal of Magnetic Resonance 152, 217-226. (2001)
        #
        
        # compute constants
        pts = []
        epsilons = []
        tanB = math . tan ( self . betaRad )
        secB = 1. / math . cos ( self . betaRad )

        A = self.sigma11 + self.sigma33 * tanB *tanB + \
            self.sigma22 * ( self.sigma33 - self.sigma11 ) / ( self.sigma22 - self.sigma33 )
        C1 = self . sigma22 * ( sigma - self . sigma33 ) / ( self .sigma22 - self .sigma33 )
        
        # There are 4 degeneracies
        #  use epsilon = +1./-1. to enscapsulate

        # check to see sign of dipole is known
        if sgn == -1:
            epsilon1Range = [-1]
        elif sgn == 1:
            epsilon1Range = [1]
        else:
            epsilon1Range = [1, -1]

        for epsilon1 in epsilon1Range :
            rad = (2. * epsilon1 * math.fabs(nu) / self . nuParallel + 1.)/ 3.
            if ( rad >= 0. ) :
                nu0 = tanB * secB * math.sqrt( rad )
                B = self . sigma33 * nu0
                C = self . sigma33 * nu0 * nu0 / ( tanB *tanB ) - sigma + C1
                for epsilon2 in [1, -1 ] :
                    for epsilon3 in [1, -1] :

                        rad = B*B - A*C
                        if rad >= 0. :
                            x = ( epsilon2*B + epsilon3*math.sqrt(rad) ) / A
                            z = epsilon2 * nu0 / tanB - tanB * x
                            if ( ( 1. - x*x - z*z ) >= -self.NumErr ) :
                                # handle numerical error
                                if ((( 1. -x*x - z*z ) >= -self.NumErr ) and \
                                    (( 1. -x*x - z*z ) <= 0. )):
                                    if math . fabs( z ) > math . fabs( x ):
                                        y = 0.
                                        nx = math . sqrt ( 1. - z*z )	
                                        if x < 0.:
                                            x = -nx
                                        else:
                                            x = nx
                                    else:
                                        y = 0.
                                        nz = math . sqrt ( 1. - x*x )	
                                        if z < 0.:
                                            z = -nz
                                        else:
                                            z = nz 
                                else:
                                    y = math . sqrt ( 1. - x*x - z*z )
                                pts . append ( Vector ( x,y,z ) )
                                epsilons . append ( [ epsilon1, epsilon2, epsilon3, 1 ] )
                                if not allDegeneracies:
                                    # just return first valid degeneracy
                                    return pts, epsilons
                                # 4th degeneracy done quickly to save time
                                pts . append ( Vector ( x,-y,z ) )
                                epsilons . append ( [ epsilon1, epsilon2, epsilon3, -1 ] )

                       
        return pts, epsilons

    def projectToFreqPlane ( self, p ) :
        sigma = self . sigma11 * p._x * p._x + self . sigma22 * p._y * p._y  + \
                self . sigma33 * p._z * p._z
        nu = self . nuParallel / 2. * \
             ( 3. * ( \
                 ( math . sin (self.betaRad) * p._x + math . cos ( self.betaRad ) * p._z ) * \
                 ( math . sin (self.betaRad) * p._x + math . cos ( self.betaRad ) * p._z ) ) - 1. )
        return sigma, nu

    def printEllipse ( self, n, negPos = 0 ) :

	delta = ( self . sigma33 - self . sigma11 ) / n
        nuPos = []
        nuNeg = []
	for i in range ( n + 1 ) :
	   sigma = self . sigma11 + i * delta
	   nuPos . append ( self.getNu( sigma ) )
	   nuNeg . append ( self.getNu( sigma, -1 ) )
        if negPos <> -1 :
            for i in range ( n + 1 ) :
                sigma = self . sigma11 + i * delta
                print sigma,  nuPos [ i ]
            for i in range ( n + 1 ) :
                sigma = self . sigma11 + ( n - i ) * delta 
                print sigma , nuNeg [ n - i ]
        if negPos == -1 :
            for i in range ( n + 1 ) :
                sigma = self . sigma11 + i * delta
                print sigma , -nuPos [ i ]
            for i in range ( n + 1 ) :
                sigma = self . sigma11 + ( n - i ) * delta 
                print sigma, -nuNeg [ n - i ]    

    def convert( self, aa ) :
        # convert 3-letter to 1-letter AA
        AAdict = { 'ALA':'A', 'ARG':'R', 'ASN':'N', 'ASP':'D', 'CYS':'C',
                   'GLN':'Q', 'GLU':'E', 'GLY':'G', 'HIS':'H', 'ILE':'I',
                   'LEU':'L', 'LYS':'K', 'MET':'M', 'PHE':'F', 'PRO':'P',
                   'SER':'S', 'THR':'T', 'TRP':'W', 'TYR':'Y', 'VAL':'V' }
        return AAdict [ aa ] 

####################################################

if __name__ == '__main__' :
    
    def cleanResidueRanges ( res ) :
	for r in res :
	    ii = string. rfind ( r, ":" )
	    if ii == -1 or ii == 0 :
		print "Error - range incorrect"
		usage ( )
	    f = int ( r [ :ii ] )
	    l = int ( r [ ( ii + 1 ): ] )
	    if ( f and l and f > l ) :
		print "Error - ranges out of order"
		usage ( ) 	
	if ( len ( res ) == 0 ) :
		res . append ( "0:0" )

    def doPowderPattern ( nmr, s, rezs, plotData, dipolarAngle, \
                          printResNumber, printResType ) :
        for i in range ( len ( nmr . Bs ) ) :
            nmr . getDipolePowder ( s, rezs, nmr . Bs [ i ] )
        if plotData:
            for i in range ( len ( nmr . sigmas ) ) :
                enu = nmr . nus [ i ] 
                if nmr . reflect :
                    enu = math . fabs ( nmr . nus [ i ]  )
		if printResNumber :
                    print nmr . numbers [ i ] , nmr . sigmas [ i ] , enu 
                elif printResType:
                    print nmr . types[i], nmr . sigmas[i], enu
		else:
                    print nmr . sigmas [ i ] , enu 
        if dipolarAngle :
            for i in range ( len ( nmr . dipangles ) ) :
                if printResNumber :
                    print nmr . numbers [ i ] , \
                          "NH <-> B: ", nmr . toDegree ( nmr . dipangles [ i ] )
                elif printResType:
                     print nmr . types [ i ] , \
                           "NH <-> B: ", nmr . toDegree ( nmr . dipangles [ i ] )
                else:
                    print "NH <-> B: ", nmr . toDegree ( nmr . dipangles [ i ] )


    def usage ( ) :
        print "Usage: " + sys . argv [ 0 ] + " [options] -i PDBfile"
        print "Performs ssNMR dipolar/cs calculations"
        print "where options are: "
        print "\t-r i:j        just use residues i to j [1:0 == first:last]"
        print "\t-c            compute C13 dipolar info [N15-H backbone is default]"
        print "\t-I            compute N15 Indole info [N15-H backbone]"
        print "\t-b            beta angle in degrees [17(NH)]"
        print "\t-n            nuParallel value [11.335]"
        print "\t-1            sigma11 principle value [30]"
        print "\t-2            sigma22 principle value [60]"
        print "\t-3            sigma33 principle value [205]"
        print "\t-a            use calculated betas per residue for NH"
        print "\t print options"
        print "\t\t-N            print residue number beside data"
        print "\t\t-T            print residue type beside data"
        print "\t\t-E            don't print ellipses"
        print "\t\t-D            don't print any NMR data"
        print "\t\t-R            don't reflect dipolar points"
        print "\t\t-P            print sig11, sig22, sig33 [PAF] per residue"
        print "\t\t-A            print dipolar angle with B field ( deg )"
        print "\t\t-Z            print spherical projection ( ie, B in PAS )"
        sys . exit ( 0 )

    if len ( sys .argv ) == 1 :
            usage () 

    try:
        opts, args = getopt.getopt(sys.argv[1:], "sEATUPINDRZab:n:1:2:3:cr:i:")
    except getopt.error, msg:
        sys.stderr.write(sys.argv[0] + ': ' + str(msg) + '\n')
        usage ( )

    # flags
    rezs = [] 
    ellipse = 1
    nc = 0
    nh = 1
    indole = 0
    plotData = 1
    reflect = 1
    uniform = 1
    betaAng = -1
    sigma11 = -1
    sigma22 = -1
    sigma33 = -1
    nupar = -1
    rawbeta = 0
    PAF = 0
    sphereProj = 0
    dipolarAngle = 0
    printResNumber = 0
    printResType = 0

    for o, a in opts :
        if o == '-r' :
            rezs . append ( a ) 
        if o == '-i':
            inFile = a     
        if o == '-E':
            ellipse = 0
        if o == '-R':
            reflect = 0
        if o == '-I':
            indole = 1
            nc = 0
            nh = 0
        if o == '-a':
            rawbeta = 1
        if o == '-D':
            plotData = 0
        if o == '-c':
            nc = 1
            nh = 0
        if o == '-Z':
            sphereProj = 1
        if o == '-h':
            nh = 1
            nc = 0
        if o == '-U':
            stats = 1
        if o == '-b':
            betaAng = float( a )
        if o == '-n':
            nupar = float( a )
        if o == '-1':
            sigma11 = float( a )
        if o == '-2':
            sigma22 = float( a )
        if o == '-3':
            sigma33 = float( a )
        if o == '-P':
            PAF = 1
        if o == '-A':
            dipolarAngle = 1
        if o == '-N':
            printResNumber = 1
        if o == '-T':
            printResType = 1
            
    # initialize NMR processor
    nmr = ssNMR ( )  
    nmr . PAF = PAF
    nmr . dipolarAngle = dipolarAngle

    if nupar <> -1 :
        if nh:
            nmr . NHnuParallel = nupar
        if nc:
            nmr . NCnuParallel = nupar
        if indole:
            nmr . IndolenuParallel = nupar
    if betaAng <> -1 :
        if nh:
            nmr . NHbetaRad = nmr . toRad ( betaAng ) 
        if nc:
            nmr . NCbetaRad = nmr . toRad ( betaAng )
        if indole:
            nmr . IndolebetaRad = nmr . toRad ( betaAng )
    if sigma11 <> -1 :
        if nh:
            nmr . NHsigma11 = sigma11
        if nc:
            nmr . NCsigma11 = sigma11
        if indole:
            nmr . Indolesigma11 = sigma11
    if sigma22 <> -1 :
        if nh:
            nmr . NHsigma22 = sigma22
        if nc:
            nmr . NCsigma22 = sigma22
        if indole:
            nmr . Indolesigma22 = sigma22
    if sigma33 <> -1 :
        if nh:
            nmr . NHsigma33 = sigma33
        if nc:
            nmr . NCsigma33 = sigma33
        if indole:
            nmr . Indolesigma33 = sigma33
            
    nmr . setReflection ( reflect )
    nmr . setBetaDetermination ( rawbeta ) 

    ## ------------  initialize PDB structure
    s = Structure ( inFile )
    s . computePeptidePlaneNormals ( ) 

    cleanResidueRanges ( rezs )
    
    if nc :
        nmr . setNC ( ) 
    if nh :
        nmr . setNH ( ) 
    if indole :
        nmr . setIndole ( )

    ## ------------  print ellipse
    if ellipse :
        nmr . printEllipse ( 250 )
        print " "
        nmr . printEllipse ( 250, -1 )
        print ""

    # default is B parallel to Z axis
    nmr . Bs . append ( Vector ( 0., 0., 1. ) ) 

    ## ------------  get powder data
    NCsigs = []
    NCnus = []
    NHsigs= []
    NHnus = []

    if nc:
        nmr . setNC ( )
        doPowderPattern ( nmr, s, rezs, plotData, dipolarAngle, \
                          printResNumber, printResType )
        NCsigs = nmr . sigmas [:]
        NCnus = nmr . nus [:]

    if nh:
        # clear it for new values
        del nmr . sigmas [:]
        del nmr . nus [:]        
        nmr . setNH ( )
        doPowderPattern ( nmr, s, rezs, plotData, dipolarAngle, \
                          printResNumber, printResType )
        NHsigs = nmr . sigmas [:]
        NHnus = nmr . nus [:]
    if indole:
        # clear it for new values
        del nmr . sigmas [:]
        del nmr . nus [:]        
        nmr . setIndole ( )
        doPowderPattern ( nmr, s, rezs, plotData, dipolarAngle, \
                          printResNumber, printResType )
        Indolesigs = nmr . sigmas [:]
        Indolenus = nmr . nus [:]

        
    if sphereProj:            
        if nh :
            nmr . setNH ( )
            for i in range ( len ( NHsigs ) ):
                p, e = nmr . getBinPAF ( NHsigs [ i ] ,
                                       math . fabs ( NHnus [ i ] ),
                                       nmr . NHbetaRad )
                #for j in range ( len ( p ) ):
                #    print p[j],e[j]
                for j in range (1):
                    print p[j][0], p[j][1], p[j][2]
                
        if nc :
            nmr . setNC ( ) 
            for i in range ( len ( NCsigs ) ):
                p, e = nmr . getBinPAF ( NCsigs [ i ] ,
                                      math . fabs ( NCnus [ i ] ),
                                      nmr . NCbetaRad )
                for j in range ( len ( p ) ):
                    print p[j],e[j]
                    
