#! /usr/bin/env python
#
#  pipath -
#
#  finds an assignment of NMR data by looking at torsion angle
#  pairs and building a graph
#  problem essentially becomes the TSP problem on an undirected graph
#

import os
import sys
import getopt
import math

from FSUPeptidePlane import PeptidePlane
from FSUPeptidePlane import Propagator
from FSUNMRBase import NMRBase
from FSUPDB import *
import FSUUtils
import FSUGraph
import FSULog

# HARD LIMITS
MAXBOUND = 10000 # prevents runaways for graphs that are invalid
MAXBOUNDINCR = 100 # prevent searching after you have found some
# tours but haven't yet reached your goal of maxTours 
NOPATHBOUNDINCR = 10 # amount to increase bound when no assgnments have been found
PATHBOUNDINCR = 5  # amount to increase bound when some assgnments have been found
# but not maxAssign, ie, getting close

class PiPath :

  #
  # class which encapsulates the control
  # algorithm - calling the propagator as necessary
  #

  def __init__ ( s, prop ) :
    s . p = prop
    s . installDir = os.environ["PIPATH_INSTALL_DIR"]
    s . aaStructureFile = s . installDir + '/lib/FSUAminoAcids.pdb'
    s . aaDict = { 'A':'ALA', 'R':'ARG', 'N':'ASN', 'D':'ASP', 'C':'CYS',
                   'Q':'GLN', 'E':'GLU', 'G':'GLY', 'H':'HIS', 'I':'ILE',
                   'L':'LEU', 'K':'LYS', 'M':'MET', 'F':'PHE', 'P':'PRO',
                   'S':'SER', 'T':'THR', 'W':'TRP', 'Y':'TYR', 'V':'VAL' }
    s . seq = ""
    s . spectrum = []
    s . ERR_MATCH = 0.1
    s . NUM_ERR = 0.00001
    s . verbose = 0
    s . correlate = 2
    s . aaDB = s . createAminoAcidDB ( )
    s . PHI = -65.
    s . PSI = -40.
    s . moveGrammian = 1
    s . torsionAngleBound = 0

  def verifySeq ( s, seq ) :
    for i in seq:
      if not s . aaDict . has_key ( string . upper ( i ) ) :
        return 0
    return 1

  def dumpData ( s, log ) :

    for i in range ( len ( s . seq ) ):
      log . write (  s . aaDict [ s . seq [ i ] ] + str (i) + '\n' )
    log . write ( "DATA:\n" )
    log . write ( "Correlation Type : " + str(s . correlate) + "D\n" )
    for i in range ( len ( s . seq ) ):
      dString =  ' ' + str ( s . spectrum [i][0] ) + \
                "\t" + str ( s . spectrum [i][1] )
      if s.spectrum[i][0] != 0. and \
         s.spectrum[i][1] != 0.:
        if s . spectrum[i][3] == 0 :
          sgndip = s . p . nmr . determineDipoleSign ( s.spectrum[i][1],
                                               s.spectrum[i][0],
                                               s . correlate )
        else:
          sgndip = s . spectrum[i][3]
        log . write ( dString + " sgn: " + str(sgndip) + '\n' )
      else:
        log . write( dString + '\n' )
    log . write ( '\n' )

  def readData ( s, file, log ) :
    seqPattern = re. compile ( r'\s*(\S+)' )
    dataPattern = re . compile ( r'\s*(\S+)\s+(\S+)\s+([+-])((\s+\d+)*)')
    dataPattern2 = re . compile ( r'\s*(\S+)\s+(\S+)((\s+\d+)*)')
    seqMatched = 0
    ofile = open ( file , 'r' )
    seq = ''
    lineno = 0
    for line in ofile . readlines ( ) :
      lineno = lineno + 1
      parseError = 0
      if lineno == 1 :
        match = seqPattern . search( line )
        if match:
          seq = match . group ( 1 )
          s . seq = seq
          if not s . verifySeq ( s . seq ) :
            print "Line ", lineno
            print "Error : invalid AA sequence specified in data file"
            sys . exit (-1)
        else:
          parseError = 1
      else:
        match = dataPattern . search ( line )
        if match :
          ( cs, dipole ) = float(match . group ( 1 )) , float(match . group ( 2 ))
         # compute B in PAF for each observed data point
          sgn = 0
          ul = []
          valueString = match . group ( 3 )
          if valueString == '+':
            sgn = 1
            valueString = match . group ( 4 )
          elif valueString == '-':
            sgn = -1
            valueString = match . group ( 4 )
          if cs == 0.0 and dipole == 0.0 :
            ipos = []
            ipos. append ( Vector (0.0, 0.0, 0.0) )
          else:
            # have to get ALL possible degeneracies here            
            ipos, eps = s . p . nmr . getBinPAF ( cs, dipole, sgn )
          if valueString :
            for i in valueString . split ( ' ' ):
              if i != '' :
                ul . append ( int ( i ) )
          data = [ cs, dipole, ipos, sgn, eps, ul ]
          s . spectrum . append ( data )
        else:
          match = dataPattern2 . search ( line )
          if match :
            ( cs, dipole ) = float(match . group ( 1 )) , float(match . group ( 2 ))
            # compute B in PAF for each observed data point
            sgn = 0
            ul = []
            valueString = match . group ( 3 )
            if cs == 0.0 and dipole == 0.0 :
              ipos = []
              ipos. append ( Vector (0.0, 0.0, 0.0) )
            else:
              sgn = s . p . nmr . determineDipoleSign ( dipole, cs, s .correlate)
              ipos, eps = s . p . nmr . getBinPAF ( cs, dipole, sgn )
            if valueString :
              for i in valueString . split ( ' ' ):
                if i != '' :
                  ul . append ( int ( i ) )
            data = [ cs, dipole, ipos, sgn, eps, ul ]
            s . spectrum . append ( data )
          else:
            parseError = 1
        if parseError:
          print "Error parsing dataFile :"
          print line
          print "    see usage \'-h\'"
          sys .exit ( -1 )
    
    if len( s . seq ) <> len ( s . spectrum ):
      print len( s .seq ) , len ( s . spectrum )
      print "Primary sequence and data set don't have equivalent length"
      sys .exit( -1 )
    if s . verbose:
      log . write( s . seq + '\n' )
      for ss in s . spectrum :
        log . write( str( ss[0] ) + ' ' + str( ss[1] ) + '\n' )
        for sd in ss[2]:
          log . write( '\t' + str( sd ) + '\n' )
        if len(ss[5]):
          log .write ( "pin this residue to position(s): " + str( ss[5]) + "\n" )
    ofile . close ( )

  def createAminoAcidDB ( s ) :

    # this reads in all 20 amino acid residues
    # structures
    # Read from a simple PDB file
    if s . verbose:
      print 'loading AA db..'
    aas = Structure ( s . aaStructureFile )
    # now verify all AAs present
    if len ( aas . residues ) != len ( s . aaDict ) :
      print "Error: amino acid db mismatch"
      sys . exit ( -1 )

    # build aaName->resIndex db
    db = {}
    for i in s . aaDict . keys() :
      match = 0
      for j in aas . residues :
        if s . aaDict [ i ] == j . name :
          match = 1
          db [ i ] = j
          continue
      if not match :
        print "Error: ", i, " not found in db"
        sys . exit ( -1 )
    if s . verbose:
      print "db verified."
    return db

  def checkHandedness ( s, t ):
    vl = t . convertEdgePathtoVertList ( )
    for j in range( len ( vl )):
      vl[j] = int( vl[j] )
    newDataOrdering = []
    for i in vl:
      data = s . spectrum [i]
      newDataOrdering . append ( ( data[0], data[1] ) )

    crosses = []
    for i in range( len( newDataOrdering ) - 2 ):

      csa, dipa = newDataOrdering[i]
      csb, dipb = newDataOrdering[i + 1]
      csc, dipc = newDataOrdering[i + 2]
      ax = csb - csa
      ay = dipb - dipa
      bx = csc - csb
      by = dipc - dipb
      
      alen = math . sqrt ( ax*ax + ay*ay )
      blen = math . sqrt ( bx*bx + by*by )
      cross = ax*by - ay*bx
      cross /= alen*blen
      cross *= len( data)
      crosses . append ( cross )


    # should be rotating in one direction
    # (usually counter-clockwise but not necessarily)
    a = crosses[ 0 ]
    bad = 0
    for c in range( 1, len( crosses)):
      if a * crosses [ c ] < 0. :
        bad = 1

    if bad :
      return 0
    else:
      if a > 0. :
        # counter - clockwise in normal ssNMR powder graph
        return 1
      else:
        return -1
    
  def sortTours ( s, tours ):
    wts = []
    for t in tours:
      wts . append ( ( t . totalWeight(), t ) )
    wts . sort ( )
    del tours[:]
    for wt, t in wts:
      tours . append ( t )

  def prune ( s, g ) :
    # run thru pins and see if there are any consecutive
    # single labels
    singles = []
    for r in range( len(s . spectrum )):
      pins = s . spectrum[r][5]
      if len( pins ) == 1:
        singles . append ( (r, pins[0]) )
    pruneEdges = []
    for sa, sb in singles:
      for ta, tb in singles:
        if sa <> ta:
          if tb == sb + 1:
            # consecutive
            pruneEdges . append ( (sa, ta ))
    for pa,pb in pruneEdges:
      log .write ( "PRUNE to perserve " + str(pa) + ' ' + str(pb) + "\n" )
      paStr = "%02d" % pa
      pbStr = "%02d" % pb
      for js in range( len( g.E )-1,-1,-1):
        ea, eb, ew = g.E[js]
        if ea == paStr and eb <> pbStr:
          del g.E[js]
          log.write( "remove " +str(ea) + "->" + str(eb) + " " + str(ew) + "\n")
        elif eb == pbStr and ea <> paStr :
          del g.E[js]
          log.write( "remove " +str(ea) + "->" + str(eb) + " " + str(ew) + "\n")
        elif ea == pbStr and eb == paStr:
          del g.E[js]
          log.write( "remove " +str(ea) + "->" + str(eb) + " " + str(ew) + "\n")

  def computeDeviation( s, phi, psi ):
    dphi = math . fabs ( phi - s . PHI )
    dpsi = math . fabs ( psi - s . PSI )
    if dphi > 180.:
      dphi = 360. - dphi
    if dpsi > 180.:
      dpsi = 360. - dpsi
    delta = math . sqrt(  dphi*dphi + dpsi*dpsi )
    return delta

  def getMininalTorsionAngle( s, i, j ):

    ipos = pipath . spectrum [i][2]
    jpos = pipath . spectrum [j][2]

    # compute all torsion angles between residue i and j
    Tors = []
    Exact = []
    Grams = []
    Eps = []
    minEps = []
    minDelta = 999.

    if ( len(ipos) == 0 or len(jpos) == 0 ) :
      log.write(  "invalid sequence %d -> %d.\n" % (i, j ))
    elif ipos[0].length() > 0. and jpos[0].length() > 0. :

      # these are valid data points
      # have to look at case of dipolar splitting
      for di in range ( 0, len( ipos ), 4 ) :
        for dj in range ( 0, len ( jpos ), 4 ):
          tors, exact, grams, eps = \
                prop . computeTorsionsByChiralities \
                ( ipos[di], jpos[dj], prop.nmr.betaRad )
          Tors . extend( tors )
          Exact . extend( exact )
          Grams . extend( grams )
          Eps . extend ( eps )
          
          phipsi = []
          minDelta = 999.
          minset = 0
          for k in range( len ( Tors )):
            phi, psi = Tors[k]
            # NOTE: only look at exact torsion computations
            if Exact[k]:
              delta = s . computeDeviation( phi, psi ) 
              if delta < minDelta :
                minDelta = delta
                minset = 1
                minEps = Eps[k]

          # now inexact - move gramian
          if not minset and s . moveGrammian:
            negGram = -999.
            for ig in range( len( Grams)):
              if Grams[ig] > negGram \
                 and Grams[ig] < 0.:
                negGram = Grams[ig]
                negIndex = ig
            log.write ( "Moving data point " + str(i) +  "->" + str( j )+
                        " gram %5.2f\n" % negGram )            
            phi, psi = Tors[negIndex]
            delta = s . computeDeviation( phi, psi ) 
            if delta < minDelta :
              minDelta = delta
              minEps = Eps[ig]
                  
    return minDelta, minEps

  def buildAssignmentGraph( s, g, degenDict ):

    for i in range ( len ( s . seq ) ):
      for j in range ( len ( s . seq ) ):
        if i <> j :
          minDelta, minEps =  s . getMininalTorsionAngle( i, j )

          if s . torsionAngleBound :
            if minDelta > s. torsionAngleBound:
              minDelta = 999.

          if minDelta <> 999.:
            # now add edge weighted with closest distance to alpha-helical
            iStr = "%02d" % i
            jStr = "%02d" % j
            g . addEdge ( iStr, jStr, int( minDelta + 0.5 ) )
            degenDict [ (iStr, jStr) ] = minEps

  def checkPinnedResidues( s, t, r, pins ):
    bad = 1
    vl = t . convertEdgePathtoVertList ( )
    for i in range( len(vl)):
      iv = int( vl[i] )
      if iv == r:
        # verify r in correct position
        for p in pins :
          if i == p :
            bad = 0
    return not bad


  def cleanExit ( s ) :
    sys . exit ( -1 )

if __name__ == '__main__' :

  def usage ( ) :
    print ""
    print "Usage: " + sys . argv [0] + " [options] -i dataFile"
    print ""
    print "Description: Find most alpha-helical assignments and structures that match"
    print "             an input PISEMA data set."
    print ""
    print "Options:"
    print "---control---"
    print "\t-M maxAssign stop after maxAssign assignments computed[100]"
    print "\t-E TAbound   enforce torsion angle deviation bound"
    print "\t-D           ensure consistent handedness"
    print "\t-B bound     upper bound for paths returns ALL paths below value"
    print "\t             NOTE: if B is unset, pipath adaptively raises bound from 0 until it "
    print "\t                   finds maxAssign assignments, setting B explicitly runs"
    print "\t                   pipath once for that value."
    print "---ssNMR values----"
    print "\t-b           ssNMR beta in degrees [17.0]"
    print "\t-1           ssNMR sigma11[30.0]"
    print "\t-2           ssNMR sigma22[60.0]"
    print "\t-3           ssNMR sigma33[205.0]"
    print "\t-n           ssNMR nuParallel[11.335]"
    print "\t             NOTE: nuParallel here is 1/2 of the maximal splitting"
    print "---output----"
    print "\t-v           verbose"
    print "\t-H           print sample input dataFile"
    print "\t-h           print this message"
    print "\t-S           do NOT create structure files, just compute paths"    
    print "\t-o           outputDir[pipath]"
    print "\t-l           logfile[outputDir/log]"
    print "\t-q           quiet mode: just write to logfile"    


    sys . exit ( -1 )

  def Help( ) :
    print ""
    print "Here is a sample input data file:"
    print ""
    print "AILGAVA"
    print "204.950787397 9.7157822365"
    print "186.244542352 11.0416800732"
    print "176.376738589 7.86203573461"
    print "196.085140792 7.6875575247"
    print "201.560020118 10.9194453097 3 4 5"
    print "177.724049152 9.9371262076"
    print "0.0 0.0"
    print ""
    print "where first line is the primary sequence and each following line represents:"
    print ""
    print "chemicalShift(ppm) dipolarCoupling(kHz) [optional enforced positions]"
    print ""
    print "NOTE: the order of the data points does not matter"
    print "as the algorithm finds the order of the most alpha-helical"
    print "match to the data. The optional enforced position arguments will"
    print "return only those assignments which agree with the resonance properly"
    print "positioned - here, where the data point (201.56, 10.92) occupies the"
    print "4th , 5th, or 6th position."
    print "The data point (0.0, 0.0) signifies a missing data point."
    print "IMPORTANT: position indices start with 0, not 1!"
    print ""
    sys . exit ( -1 )


  try:
    opts, args = getopt.getopt(sys.argv[1:], "GhDHSqvl:i:o:B:M:b:n:1:2:3:E:")
  except getopt.error, msg:
    sys.stderr.write(sys.argv[0] + ': ' + str(msg) + '\n')
    usage ( )

  
  outputDir = "pipath"
  verbose = 0
  debug = 0
  bound = 0
  torsions = 0
  logfile = "log"
  createStructs = 1
  tee = 1
  maxTours = 100
  dataFile = ""
  beta = 17.0
  sigma11 = 30.
  sigma22 = 60.
  sigma33 = 205.
  nuParallel = 11.335
  sampleHelp = 0
  checkHandedness = 0
  torsionAngleBound = 0
  adaptive = 1

  verifyPath = 0
  degenDict = {}

  for o, a in opts :
    if o == '-o' :
      outputDir = a
    if o == '-H' :
      sampleHelp = 1
    if o == '-v':
      verbose = 1
    if o == '-h':
      usage ()
    if o == '-B':
      bound = int ( a )
      adaptive = 0
    if o == '-S':
      createStructs = 0
    if o == '-l':
      logfile = a
    if o == '-q':
      tee = 0
    if o == '-i':
      dataFile = a
    if o == '-n':
      nuParallel = float( a )
    if o == '-1':
      sigma11 = float( a )
    if o == '-2':
      sigma22 = float( a )
    if o == '-3':
      sigma33 = float( a )
    if o == '-b':
      beta = float( a )
    if o == '-M':
      maxTours = int(a)
    if o == '-D':
      checkHandedness = 1
    if o == '-E':
      torsionAngleBound = int( a )

  if sampleHelp:
    Help()

  if not len(dataFile ):
    usage()


  utils = FSUUtils.Utils()      
  utils . makedirs( outputDir, 1 )
  STRUCTBUILDER = "structbuilder.py"

  log = FSULog . Log( open( outputDir + "/" + logfile, "w" ))
  if tee:
    log . split = 1

  mainTimer = log . startTimer ( ) 

  nmr = NMRBase ( )
  nmr . betaRad =  nmr.toRad( beta )
  nmr . nuParallel = nuParallel
  nmr . sigma11 = sigma11
  nmr . sigma22 = sigma22
  nmr . sigma33 = sigma33     
  nmr . reflect = 0

  prop = Propagator ( nmr )

  pipath = PiPath ( prop )
  pipath . verbose = verbose
  pipath . readData ( dataFile, log )
  pipath . torsionAngleBound = torsionAngleBound

  if verbose:
    pipath . dumpData ( log ) 
    log . write ( "\nNMR Constants:\n" )
    log . write ( "\tbeta(rad): %7.3f\n" % nmr.betaRad )
    log . write ( "\tnuParallel: %7.3f\n" % nmr.nuParallel )
    log . write ( "\tsigma11: %7.3f\n" % nmr.sigma11 )
    log . write ( "\tsigma22: %7.3f\n" % nmr.sigma22 )
    log . write ( "\tsigma33: %7.3f\n\n" % nmr.sigma33 )        

  tsp = FSUGraph.TSP ( )

  g = FSUGraph.Graph()
  for i in range ( len ( pipath.seq ) ):
    iStr = "%02d" % i
    g . addVertex ( iStr )

  # build data graph --------------------------
  log . write( "Building Graph...\n" )
  log . indent ()

  pipath . buildAssignmentGraph( g, degenDict )
  
  log . unindent()

  # add non-data residue edges ( if any )
  for i in range ( len ( pipath.seq ) ):
    ipos = pipath . spectrum [i][2]
    if len(ipos) and ipos[0].length() == 0:
      # add an Edge both ways to every vertex with a weight of 1
      for j in range( len( pipath.seq )):
        if i <> j :
          iStr = "%02d" % i
          jStr = "%02d" % j
          g . addEdge ( iStr, jStr, 1)
          g . addEdge ( jStr, iStr, 1)

  # check for pruneable edges ( consecutive labels )
  pipath . prune ( g )
  g . saveToFile( outputDir + "/graph.out" )

  # find tours --------------------------------------
  log.write( "\tE = " + str(len ( g.E )) + ' w = ' +str(g. totalWeight()) + ' ' + \
             "%7.2f" % (float(g.totalWeight())/float(len(g.E))) + ' ' + \
             "%7.2f" % (float(g.totalWeight())/float(len(g.E))*float(len(g.V))) + "\n\n" )
  log . indent()
  found = 0
  tCount = 0

  # print stats on edges
  log . write( "Edge Stats:\n" )
  for i in range( len( g.V )):
    ws = []
    for j in range( len( g.E )):
      a, b, w = g . E[j]
      if int(a) == i or int(b) ==i :
        ws . append ( w )
    avg, sd, min, max = utils . getStats( ws )
    log . write( str( i) +  " avg wgt: " + str( avg) + " +/- " + str(sd) + \
                 " of " + str( len( ws ) ) + " edges\n" )

  # BEGIN Path finding - 
  boundStart = 0
  while not found:
    tsp . reset ( )
    tsp . justOne = 0
    tsp . maxTours = maxTours
    log.write( ( "Trying to find %d min paths of cost " % maxTours ) \
               + str(bound) + "\n")        
    tourTimer   = log . startTimer ( )

    tsp . findMinPath ( g, bound )     
    tCount += 1
      
    found = len( tsp . tours )
    if found:
      log.write( "  Found " + str(found) + " paths.\n" )
    else:
      log.write( "No paths found.\n" )

    if adaptive:
      if not found:
        # no paths found - increase bounds
        bound += NOPATHBOUNDINCR
        log.write( "Increasing bound to %d\n" % bound )
        if bound > MAXBOUND :
          found = 1
      else:
        # path found
        # expand to make sure we look at maxTours
        if found < maxTours:      
          bound += PATHBOUNDINCR
          log.write( "\tIncreasing bound to %d\n" % bound )
          found = 0
          if not boundStart:
            boundStart = bound
          else:
            if (bound - boundStart) > MAXBOUNDINCR :
              log . write("\t Runaway bound - investigate MAXBOUNDTOURS" )
              log . write("\n")
              found = 1
        else:
          log.write("Exiting...\n" )
          log.write("\n")
          found = 1
    else:
      found = 1

    log . stopTimer( tourTimer )
    # END Path finding

  if len ( tsp.tours) < maxTours:
    log . write( \
      "Only found %d paths with bound %d - try increasing bound.\n" % (len(tsp.tours), bound))
    log.write("Exiting...\n" )
    log.write("\n")
    log . unindent()
    log . stopTimer( mainTimer )
    log . close()
    pipath . cleanExit ( ) 

  # post-process tours -----------------------------
  log . write ( "\npost-filtering tours....\n\n")
  badTours = []
  for i in range ( len ( tsp . tours ) ):
    t = tsp.tours[i]
    tourStr = "%04d" % i
    if checkHandedness:
      if not pipath . checkHandedness ( t ) :
        badTours . append ( i )
        log . write ( "%s : no handedness - rejecting...\n" % tourStr )  

  if len( badTours ) :
    for i in range( len( badTours) -1, -1, -1):
      del tsp . tours[ badTours[i]]
    del badTours[:]

  # checking for pinning residues
  for r in range( len( pipath . spectrum ) ):
    pins = pipath . spectrum[r][5]
    if len( pins ):
      for i in range ( len ( tsp . tours ) ):
        t = tsp.tours[i]
        tourStr = "%04d" % i
        if not pipath . checkPinnedResidues( t, r, pins ):
          badTours . append ( i )
          log . write ( "%s : resonance out of position - rejecting...\n" % tourStr ) 
      if len( badTours ) :
        for i in range( len( badTours) -1, -1, -1):
          del tsp . tours[ badTours[i]]
        del badTours[:]

  # analyze tours -------------------------------
  pipath . sortTours ( tsp . tours ) 
  for i in range(len( tsp . tours )):
    t = tsp.tours[i]
    vl = t . convertEdgePathtoVertList ( )
    ivl = []
    for j in range( len(vl) ):
      ivl . append( int (vl[j]))
    tourStr = "%04d" % i
    log.write( "PATH " + str( tourStr ) + " " + str(ivl) + " " + str( t.totalWeight()) + "\n" )
    if createStructs:
      structFile = outputDir + "/" + tourStr + ".dat"
      sfile = open ( structFile, "w" )
      for j in range( len(ivl)):
        sfile.write( pipath.seq[j] + " " + \
                     str( pipath.spectrum[ivl[j]][0]) + " " + \
                     str( pipath.spectrum[ivl[j]][1]) + '\n')
      sfile . close ( )

  log . stopTimer( tourTimer )
  log . unindent()

  nTours = []  

  if createStructs:

    log . write ( "\n\nBuilding pdbs...\n" )
    log . indent()
    contTimer = log . startTimer( ) 
    command = STRUCTBUILDER + ' -b ' + str( beta ) + ' -n ' + str( nuParallel ) + \
              ' -1 ' + str( sigma11 ) + ' -2 ' + str( sigma22 ) + ' -3 ' + str( sigma33 ) + \
              ' -o '
  
    for i in range ( len ( tsp . tours ) ):

      t = tsp.tours[i]
      tourStr = "%04d" % i
      structFile = outputDir + "/" + tourStr + ".dat"
      outFile = outputDir + "/" + tourStr

      buildCommand = command + outFile + ' -i ' + structFile
      p = os. popen ( buildCommand, 'r' )
      line = p.readline()
      p . close()
      devlist = string . split ( line ) 
      structScore = int( devlist[0] )
      
      vl = t . convertEdgePathtoVertList ( )
      for j in range( len ( vl )):
        vl[j] = int( vl[j] )
      nTours . append ( ( structScore, vl, i, t . totalWeight()) )
      log . write( "built " + outFile + " " + log . now() + "\n")

    nTours . sort()
    for itour in range( len( nTours )):
      dev, t, pathNum, otw = nTours[itour]
      log . write( " STRUCT %04d " % pathNum + str(t) + "%4d" % otw + " %4d\n" % dev) 

    log . stopTimer( contTimer )
    log .unindent()

  log . unindent()
  log . stopTimer( mainTimer )

  log . close()
  pipath . cleanExit ( ) 
