static char SccsId[] = "@(#)hydrogen.c	1.1 06/24/97 polymer";

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
 
#include <tcl.h>
#include <tk.h>
#include <togl.h>
#include "polymerl.h"

extern struct PLargs arg;
extern float *x, *y, *z;
extern int Color_Revert;
static GLfloat yellow_color[4] = { 1.0, 1.0, 0.0 };
static GLfloat green_color[4] = { 0.0, 1.0, 0.0 };
static GLfloat blue_color[4] = { 0.0, 1.0, 1.0 };
static GLfloat red_color[4] = { 1.0, 0.0, 0.0 };
static GLfloat black_color[4] = { 0.0, 0.0, 0.0 };
static GLfloat white_color[4] = { 1.0, 1.0, 1.0 };
static GLfloat silver_color[4] = { 0.9, 0.9, 0.8 };

/********************************************/
/*drawH_bonds:
  draw hydrogen atoms and H-C bonds as
  cylinders and dodecahedrons
  using OpenGL                              */
/********************************************/
void
drawH_bonds(length)
{
  int i, side;
  GLfloat ps = 5;
  GLdouble ct[3], delta[3];
  float *vx, *vy, *vz;
  float *ux, *uy, *uz;

  if (((vx = (float *) malloc((length+3)*sizeof(float))) == NULL) ||
      ((vy = (float *) malloc((length+3)*sizeof(float))) == NULL) ||
      ((vz = (float *) malloc((length+3)*sizeof(float))) == NULL)) {
             error(" can not grant malloc request.");
             exit(1);
  }
  if (((ux = (float *) malloc((length+3)*sizeof(float))) == NULL) ||
      ((uy = (float *) malloc((length+3)*sizeof(float))) == NULL) ||
      ((uz = (float *) malloc((length+3)*sizeof(float))) == NULL)) {
             error(" can not grant malloc request.");
             exit(1);
  }

  render_end_Hs(length);
  get_Hs(length, vx, vy, vz, ux, uy, uz);

  if (arg.mode >= 2) 
     glEnable (GL_LIGHTING);
  else
     glDisable (GL_LIGHTING);

  if ( Color_Revert )
     glColor3fv (black_color);
  else
     glColor3fv (white_color);

  /* points for H atoms */
  for (i = 1; i < length; i++) {
     ct[0] = ux[i];
     ct[1] = uy[i];
     ct[2] = uz[i];
     if (arg.mode >= 2) /* flat or smooth shading */
	dodecahedron(ct, radius, GL_TRIANGLE_FAN);
     else		/* wireframe */
        dodecahedron(ct, radius, GL_LINE_LOOP);
     ct[0] = vx[i];
     ct[1] = vy[i];
     ct[2] = vz[i];
     if (arg.mode >= 2) /* flat or smooth shading */
        dodecahedron(ct, radius, GL_TRIANGLE_FAN);
     else
        dodecahedron(ct, radius, GL_LINE_LOOP);
  }
  /* bonds */
  glColor3fv (silver_color);
  for (i = 1; i < length; i++) {
     ct[0] = x[i];
     ct[1] = y[i];
     ct[2] = z[i];
     delta[0] = ux[i] - x[i];
     delta[1] = uy[i] - y[i];
     delta[2] = uz[i] - z[i];
     if (arg.mode >= 2) /* flat or smooth shading */
        mycylinder (ct, radius/2, 4, delta, GL_TRIANGLE_STRIP);
     else
        mycylinder (ct, radius/2, 4, delta, GL_LINE_STRIP);
     delta[0] = vx[i] - x[i];
     delta[1] = vy[i] - y[i];
     delta[2] = vz[i] - z[i];
     if (arg.mode >= 2) /* flat or smooth shading */
        mycylinder (ct, radius/2, 4, delta, GL_TRIANGLE_STRIP);
     else
        mycylinder (ct, radius/2, 4, delta, GL_LINE_STRIP);
  }
  if (arg.mode >= 2)
     glDisable (GL_LIGHTING);
  else
     glEnable (GL_LIGHTING);

  free(vx); free(vy); free(vz);
  free(ux); free(uy); free(uz);

  return;
}
/************************************************/
/*render_end_Hs:
  draw the two end hydrogen atoms and H-C bonds
  using OpenGL                                  */
/************************************************/
void
render_end_Hs(int length) 
{
  int i, j, l;
  GLfloat ps = 4;
  GLdouble Hx[3], Hy[3], Hz[3];
  GLdouble ct[3], delta[3];

  glDisable (GL_LIGHTING);

  /* H atoms at two ends */

  for (j = 0; j < 2; j++) {

     l = j*length;
     get_End_Hs(l, Hx, Hy, Hz);

     if (arg.mode == 0) {
        glPointSize(ps/2);
        glColor3fv (yellow_color);
        glBegin (GL_POINTS);
          for (i = 0; i < 3; i++)
             glVertex3f (Hx[i], Hy[i], Hz[i]);
        glEnd ();

        /* lines for H bonds */
        if ( Color_Revert )
           glColor3fv (black_color);
        else
           glColor3fv (white_color);

        for (i = 0; i < 3; i++) {
           glBegin (GL_LINES);
             glVertex3f (Hx[i], Hy[i], Hz[i]);
             glVertex3f (x[l], y[l], z[l]);
           glEnd ();
        }
     }
     else {
        glColor3fv (silver_color);
        for (i = 0; i < 3; i++) {
           ct[0] = Hx[i];
           ct[1] = Hy[i];
           ct[2] = Hz[i];
           if (arg.mode >= 2) /* flat or smooth shading */
               dodecahedron(ct, radius, GL_TRIANGLE_FAN);
           else               /* wireframe */
               dodecahedron(ct, radius, GL_LINE_LOOP);
           delta[0] = x[l] - Hx[i];
           delta[1] = y[l] - Hy[i];
           delta[2] = z[l] - Hz[i];
           if (arg.mode >= 2) /* flat or smooth shading */
               mycylinder (ct, radius/2, 4, delta, GL_TRIANGLE_STRIP);
           else
               mycylinder (ct, radius/2, 4, delta, GL_LINE_STRIP);
        }
     }
  }

  glEnable (GL_LIGHTING);

  return;
}

/********************************************/
/*get_End_Hs:
  get coordinates of two end hydrogen atoms
/********************************************/
void
get_End_Hs(int end, GLdouble Hx[3], GLdouble Hy[3], GLdouble Hz[3]) 
{
   int i, u[3], next[3];
   
   if (end) {
      u[0] = x[end] - x[end-1];
      u[1] = y[end] - y[end-1];
      u[2] = z[end] - z[end-1];
   }
   else {
      u[0] = x[0] - x[1];
      u[1] = y[0] - y[1];
      u[2] = z[0] - z[1];
   }

   get_next(get_number(u), next);

   for (i = 0; i < 3; i++) {
      get_vector(next[i], u);
      Hx[i] = u[0]*0.5+x[end];
      Hy[i] = u[1]*0.5+y[end];
      Hz[i] = u[2]*0.5+z[end];
   }
   return; 
}

/***********************************************/
/*drawH_line:
  draw hydrogen atoms as line segements and dots
  using OpenGL                                 */
/***********************************************/
void
drawH_line(length)
{
  int i, side;
  GLfloat ps = 4;
  float *vx, *vy, *vz;
  float *ux, *uy, *uz;
 
  if (((vx = (float *) malloc((length+3)*sizeof(float))) == NULL) ||
      ((vy = (float *) malloc((length+3)*sizeof(float))) == NULL) ||
      ((vz = (float *) malloc((length+3)*sizeof(float))) == NULL)) {
             error(" can not grant malloc request.");
             exit(1);
  }
  if (((ux = (float *) malloc((length+3)*sizeof(float))) == NULL) ||
      ((uy = (float *) malloc((length+3)*sizeof(float))) == NULL) ||
      ((uz = (float *) malloc((length+3)*sizeof(float))) == NULL)) {
             error(" can not grant malloc request.");
             exit(1);
  }

  render_end_Hs(length);
  get_Hs(length, vx, vy, vz, ux, uy, uz);

  glDisable (GL_LIGHTING);

  /* points for H atoms */
  glPointSize(ps/2);
  glColor3fv (yellow_color);
  glBegin (GL_POINTS);
    for (i = 1; i < length; i++) {
       glVertex3f (vx[i], vy[i], vz[i]);
       glVertex3f (ux[i], uy[i], uz[i]);
    }
  glEnd ();

  /* lines for H bonds */
  if ( Color_Revert )
     glColor3fv (black_color);
  else
     glColor3fv (white_color);
  for (i = 1; i < length; i++) {
     glBegin (GL_LINES);
       glVertex3f (ux[i], uy[i], uz[i]);
       glVertex3f (x[i], y[i], z[i]);
       glVertex3f (x[i], y[i], z[i]);
       glVertex3f (vx[i], vy[i], vz[i]);
     glEnd ();
  }

  glEnable (GL_LIGHTING);
 
  free(vx); free(vy); free(vz);
  free(ux); free(uy); free(uz);

  return;
} /* drawH_line */

/***********************************************/
/*get_Hs:
  get coordinates of the hydrogen atoms        */
/***********************************************/
void
get_Hs(int length, float *vx, float *vy, float *vz, 
       float *ux, float *uy, float *uz)
{
   int i, j, b_num;
   int v[3], u[3], next[3];
   for (i = 1; i < length; i++) {
      u[0] = x[i] - x[i-1];
      u[1] = y[i] - y[i-1];
      u[2] = z[i] - z[i-1];

      b_num = get_number(u);
      get_next(b_num, next);

      v[0] = x[i+1] - x[i];
      v[1] = y[i+1] - y[i];
      v[2] = z[i+1] - z[i];

      b_num = get_number(v);

      if (next[0] == b_num) {
         get_vector(next[1], u);
         get_vector(next[2], v);
      }
      else if (next[1] == b_num) {
         get_vector(next[0], u);
         get_vector(next[2], v);
      }
      else {
         get_vector(next[0], u);
         get_vector(next[1], v);
      }
            
      vx[i] = x[i] + v[0]*sqrt(2.0)/2;
      vy[i] = y[i] + v[1]*0.5;
      vz[i] = z[i] + v[2]*sqrt(2.0)/2;
      ux[i] = x[i] + u[0]*sqrt(2.0)/2;
      uy[i] = y[i] + u[1]*0.5;
      uz[i] = z[i] + u[2]*sqrt(2.0)/2;
   }
   return;
}

/*
get_next: give the numbers of the three 
   possible following vectors
*/
void get_next(int b, int* next)
{
  switch (b) {
      case 1:
         next[0] = 3; next[1] = 5; next[2] = 7;
         return;
      case 2:
         next[0] = 4; next[1] = 5; next[2] = 7;
         return;
      case 3:
         next[0] = 1; next[1] = 6; next[2] = 8;
         return;
      case 4:
         next[0] = 2; next[1] = 6; next[2] = 8;
         return;
      case 5:
         next[0] = 1; next[1] = 2; next[2] = 6;
         return;
      case 6:
         next[0] = 3; next[1] = 4; next[2] = 5;
         return;
      case 7:
         next[0] = 1; next[1] = 2; next[2] = 8;
         return;
      case 8:
         next[0] = 3; next[1] = 4; next[2] = 7;
         return;
   }
}
/*
  get_number: give the number of the vector
*/
int get_number(int* v)
{
   if (v[0] == 0) {
      if (v[1] == 1) {
         if (v[2] == 1)
            return 1;
         else
            return 2;
      }
      else if (v[2] == 1)
          return 3;
      else
          return 4;
   }
   else if (v[0] == 1) {
      if (v[1] == 1)
	  return 5;
      else
	  return 6;
   }
   else if (v[1] == 1) 
      return 7;
   else              
      return 8;
}
/*
  get_vector:  give the vector with number b
*/
void get_vector(int b, int* v) 
{
   switch (b) {
      case 1:
         v[0] = 0; v[1] = 1; v[2] = 1;
         return;
      case 2:
         v[0] = 0; v[1] = 1; v[2] = -1;
         return;
      case 3:
         v[0] = 0; v[1] = -1; v[2] = 1;
         return;
      case 4:
         v[0] = 0; v[1] = -1; v[2] = -1;
         return;
      case 5:
         v[0] = 1; v[1] = 1; v[2] = 0;
         return;
      case 6:
         v[0] = 1; v[1] = -1; v[2] = 0;
         return;
      case 7:
         v[0] = -1; v[1] = 1; v[2] = 0;
         return;
      case 8:
         v[0] = -1; v[1] = -1; v[2] = 0;
         return;
   }
}
