static char SccsId[] = "@(#)lab.c	1.1 06/24/97  polymer";

/*    
**  PolymerLibV1.0 - the PolymerGenerator 
**
**  Copyright (C) 1997 S. F. Bellenot (bellenot@math.fsu.edu)
**                   & Z.-H. Duan (zduan@math.fsu.edu)
**
**  This software is shareware!
**  Read the file "License" for further information.
*/

/* The Polymer-Laboratory, a Tcl/Tk - OpenGL GUI for PolymerLib */
/* C-part */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <tcl.h>
#include <tk.h>
#include <togl.h>

#include "polymerl.h"

/********************************************/
/* Global Variables: 		            
x,y,z --- path of a random walk; 
arg --- a structure defining parameters 
    (temperature; seed; length; M; Y; Z;
    lattice -- (0-Cubic, 1-diamond);
    mode -- rendering mode (0-line, 1-wireframe, 
    2-flat, 3-smooth))
width, height --- size of 3D-View window
type -- polyethylene type
     (0-single chain, 1-amorphous region)
clip_flag -- rendering parameter
          (0-set clip off, 1-set clip on)
coordsys -- rendering parameter
         (0-do not draw coord sys, 1-draw coord sys)
				   	    */ 
/********************************************/
float *x, *y, *z;

struct PLargs arg;

static int polymer, width, height;

static GLfloat Grotx, Groty, Grotz;
static GLfloat Gmovx, Gmovy, Gmovz;
static GLfloat Gscale;
static GLfloat light_pos1[4];
static GLfloat polymer_color[4]; 

static int doubleside_flag;
static int clip_flag, Loop, Tie, FreeEnd;
static int coordsys = 1;
int Color_Revert = 0;

/********************************************/

/********************************************/
/*init_global:
  set initial values of global variables    */
/********************************************/
void
init_global (struct Togl *togl)
{
  polymer = 0;

  Grotx = 0.0, Groty = 0.0, Grotz = 0.0;
  Gmovx = -4.0, Gmovy = 0.0, Gmovz = 0.0;

  Gscale = 0.3;

  light_pos1[0] = 0.0;   light_pos1[1] = 0.0;
  light_pos1[2] = 100.0; light_pos1[3] = 0.0;

  polymer_color[0] = 0.9;
  polymer_color[1] = 0.9;
  polymer_color[2] = 0.8;
  polymer_color[3] = 1.0;

  clip_flag = 1; coordsys = 1;

  width = Togl_Width (togl);
  height = Togl_Height (togl);

  arg.ch2 = 0;      /* show carbons only*/
  arg.type = 1;      /* amorphous region*/
  arg.lattice = 1;   /* diamond lattice */
  arg.mode = 0;	     /* Line rendering mode */
  Loop = 1;          /* show loops */
  Tie = 1;           /* show ties */
  FreeEnd = 1;       /* show free ends */
}

/********************************************/
/*drawpolymer:
  draws a polymer in different modes,
  using OpenGL                              		 */
/********************************************/
void
drawpolymer (void)
{
    arg.x_init = 0; arg.y_init = 0; arg.z_init = 0;

    if ( z != NULL ) {
	free ( z );
	z = NULL;
    }
    if ( y != NULL ) {
	free ( y );
	y = NULL;
    }
    if ( x != NULL ) {
	free ( x );
	x = NULL;
    }
    if (((x = (float *) malloc((arg.max_length+3)*sizeof(float))) == NULL) ||
        ((y = (float *) malloc((arg.max_length+3)*sizeof(float))) == NULL) ||
        ((z = (float *) malloc((arg.max_length+3)*sizeof(float))) == NULL)) {
        error(" can not grant malloc request.");
        exit(1);
    }
    if (arg.type)
        drawamorphous_region(clip_flag, Loop, Tie, FreeEnd, doubleside_flag); 
    else {
        if (arg.mode == 0)   	/* lines */
            drawpolymerchain();
        else if (arg.mode == 1)   /* wireframes */
            drawpolymerframes();
        else if (arg.mode == 2 || arg.mode == 3)   /* flat/smooth */
            drawpolymerbonds();
        else
            fprintf(stderr, "NO %d mode.\n", arg.mode);
    }       
    return;
}			/* drawpolymer */

/********************************************/
/*drawcoordsys:
  draws a coordinate system 
  using OpenGL                              */
/********************************************/
void
drawcoordsys (void)
{
  if (coordsys > 0)
    {
      glDisable (GL_LIGHTING);
      glColor3f(1.0, 0.0, 0.0);
      glBegin (GL_LINES);
      glVertex3f (0.0, 0.0, 0.0);
      glVertex3f (1.0, 0.0, 0.0);
      glVertex3f (0.0, 0.0, 0.0);
      glVertex3f (0.0, 1.0, 0.0);
      glVertex3f (0.0, 0.0, 0.0);
      glVertex3f (0.0, 0.0, 1.0);
      glEnd ();
      glEnable (GL_LIGHTING);
    }

  return;
} /* drawcoordsys */

/********************************************/
/*create_cb:
 Togl Callback, called on creation
                                            */
/********************************************/
void
create_cb (struct Togl *togl)
{
  init_global(togl);

  glEnable (GL_DEPTH_TEST);
  glEnable (GL_NORMALIZE);
  glShadeModel (GL_FLAT);

  glLightfv (GL_LIGHT0, GL_POSITION, light_pos1);
  glEnable (GL_LIGHT0);

  glEnable (GL_LIGHTING);

  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, polymer_color);
  glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, 1.0);

  return;
} /* create_cb */

/********************************************/
/*reshape_cb:
 Togl Callback, called when window is 
 exposed or reshaped                        */
/********************************************/
void
reshape_cb (struct Togl *togl)
{
  float aspect = (float) width / (float) height;

  width = Togl_Width (togl);
  height = Togl_Height (togl);

  glViewport (0, 0, width, height);

  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  glFrustum (-aspect, aspect, -1.0, 1.0, 1.0, 100.0);

  glMatrixMode (GL_MODELVIEW);

  return;
} /* reshape_cb */


/********************************************/
/*display_cb:
 Togl Callback, displays the polymer
                                            */
/********************************************/
void
display_cb (struct Togl *togl)
{

  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glLightfv (GL_LIGHT0, GL_POSITION, light_pos1);

  if (polymer > 1) 
     Gscale = 0.3;

  glPushMatrix ();
  glTranslatef (Gmovx, Gmovy, Gmovz); 
  glTranslatef (0.0, 0.0, -10.0); 
  glRotatef (Grotx, 1.0, 0.0, 0.0);
  glRotatef (Groty, 0.0, 1.0, 0.0);
  glRotatef (Grotz, 0.0, 0.0, 1.0);
  glScalef (Gscale, Gscale, Gscale);

  switch (polymer) {
    case 1:
        drawpolymer(); 	
	break;
    case 2:
 	drawcubic_lattice();	
	break;
    case 3:
	drawdiamond_lattice();
	break;
    case 4:
	drawstates(0);
	break;
    case 5:
	drawstates(1);
	break;
    case 6:
	drawstates(2);
	break;
  }

  drawcoordsys ();

  glPopMatrix ();

  Togl_SwapBuffers (togl);

  return;
} /* display_cb */

/********************************************/
/*setmode_cb:
 Togl Callback, set's rendering mode
                                            */
/********************************************/
int
setmode_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName setMode 1|2|3\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  resetpara ();

  arg.mode = atoi (argv[2]);

  if (arg.mode == 3)
    glShadeModel (GL_SMOOTH);
  else
    glShadeModel (GL_FLAT);

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* setmode_cb */

/********************************************/
/*colorrevert_cb:
     Togl Callback             		    */
/********************************************/
int
colorrevert_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 2)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName drawSys\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  if (Color_Revert)
     Color_Revert = 0;
  else
     Color_Revert = 1;

  if ( Color_Revert ) 
     glClearColor (1.0, 1.0, 1.0, 1.0);
  else
     glClearColor (0.0, 0.0, 0.0, 1.0);


  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* colorrevert_cb */

/********************************************/
/*setclip_cb:
 Togl Callback, toggles turning on/off the
 clip                          		    */
/********************************************/
int
setclip_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 2)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName drawSys\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  if (clip_flag == 1)
    clip_flag = 0;
  else
    clip_flag = 1;

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* clip_on_off */

/********************************************/
/*teaching_aid_cb: Togl Callback, 	    */
/********************************************/
int
teaching_aid_cb (struct Togl *togl, int argc, char *argv[])
{
  int tmp;
  Tcl_Interp *interp = Togl_Interp (togl);
 
  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
                     "wrong # args: should be \"pathName drawSys\"",
                     TCL_STATIC);
      return TCL_ERROR;
    }
 
  polymer = atoi (argv[2]) + 1;

  Togl_PostRedisplay (togl);
  
  return TCL_OK;
} /* teaching-aids */

/********************************************/
/*setshow_cb:
 Togl Callback, toggles choose to show loops,
  ties, free end or all of them.            */
/********************************************/
int
setshow_cb (struct Togl *togl, int argc, char *argv[])
{
  int tmp;
  Tcl_Interp *interp = Togl_Interp (togl);
 
  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
                     "wrong # args: should be \"pathName drawSys\"",
                     TCL_STATIC);
      return TCL_ERROR;
    }

  tmp = atoi (argv[2]);
  if (tmp == 1) {
    Loop = 1;
    Tie = 1;
    FreeEnd = 1;
  }
  else if (tmp == 2) {
    Loop = 1;
    Tie = 0;
    FreeEnd = 0;
  }
  else if (tmp == 3) {
    Loop = 0;
    Tie = 1;
    FreeEnd = 0;
  }
  else {
    Loop = 0;
    Tie = 0;
    FreeEnd = 1;
  }
 
  Togl_PostRedisplay (togl);
 
  return TCL_OK;
} /* setshow_cb */

/********************************************/
/*setdoubleside_cb:
 Togl Callback, toggles turning on/off the
 doubleside                                 */
/********************************************/
int
setdoubleside_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);
 
  /* error checking */
  if (argc != 2)
    {
      Tcl_SetResult (interp,
                     "wrong # args: should be \"pathName drawSys\"",
                     TCL_STATIC);
      return TCL_ERROR;
    }
 
  if (doubleside_flag == 1)
    doubleside_flag = 0;
  else
    doubleside_flag = 1;
 
  Togl_PostRedisplay (togl);
 
  return TCL_OK;
} /* doubleside_on_off */

/********************************************/
/*show_H_cb:
 Togl Callback, toggles */
/********************************************/
int
show_H_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);
 
  /* error checking */
  if (argc != 2)
    {
      Tcl_SetResult (interp,
                     "wrong # args: should be \"pathName drawSys\"",
                     TCL_STATIC);
      return TCL_ERROR;
    }

  Togl_PostRedisplay (togl);
 
  return TCL_OK;
} /* show_H_cb */

/********************************************/
/*drawsys_cb:
 Togl Callback, toggles drawing of the
 coordinate system                          */
/********************************************/
int
drawsys_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 2)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName drawSys\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  if (coordsys == 1)
    coordsys = 0;
  else
    coordsys = 1;

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* drawsys_cb */



/********************************************/
/*setXrot_cb:
 Togl Callback, rotate display
                                            */
/********************************************/
int
setXrot_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName setXrot ?angle?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  Grotx = atof (argv[2]);

  Togl_PostRedisplay (togl);

  /* Let result string equal value */
  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* setXrot_cb */


/********************************************/
/*setYrot_cb:
 Togl Callback, rotate display
                                            */
/********************************************/
int
setYrot_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName setYrot ?angle?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  Groty = atof (argv[2]);

  Togl_PostRedisplay (togl);

  /* Let result string equal value */
  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* setYrot_cb */


/********************************************/
/*setZrot_cb:
 Togl Callback, rotate display
                                            */
/********************************************/
int
setZrot_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName setZrot ?angle?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }


  Grotz = Grotz + atof (argv[2]);
  if (Grotz > 360)
    Grotz -= 360;
  if (Grotz < 360)
    Grotz += 360;

  Togl_PostRedisplay (togl);

  /* Let result string equal value */
  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* setZrot_cb */


/********************************************/
/*zoomin_cb:
 Togl Callback, zoom display
                                            */
/********************************************/
int
zoomin_cb (struct Togl *togl, int argc, char *argv[])
{

  Gscale = Gscale + (Gscale / 10.0);

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* zoomin_cb */


/********************************************/
/*zoomout_cb:
 Togl Callback, zoom display
                                            */
/********************************************/
int
zoomout_cb (struct Togl *togl, int argc, char *argv[])
{

  Gscale = Gscale - (Gscale / 10.0);

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* zoomout_cb */


/********************************************/
/*movex_cb:
 Togl Callback, move polymer
                                            */
/********************************************/
int
movex_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);
 
  Gmovx = Gmovx + atof (argv[2]);
 
  Togl_PostRedisplay (togl);
 
  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* movex_cb */
 

/********************************************/
/*movey_cb:
 Togl Callback, move polymer
                                            */
/********************************************/
int
movey_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);
 
  Gmovy = Gmovy + atof (argv[2]);
 
  Togl_PostRedisplay (togl);
 
  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* movey_cb */
 

/********************************************/
/*movez_cb:
 Togl Callback, move polymer
                                            */
/********************************************/
int
movez_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName moveZ ?value?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  Gmovz = Gmovz + atof (argv[2]);

  Togl_PostRedisplay (togl);

  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* movez_cb */


/********************************************/
/*resetpara:
  	reset all parameters
                                            */
/********************************************/
void 
resetpara () 
{
  Grotx = 0.0;
  Groty = 0.0;
  Grotz = 0.0;
  Gmovy = 0.0;
  Gmovz = 0.0;
  if (arg.type) {  /* amorphous region */
     Gscale = 0.3;
     Gmovx = -4.0;
  }
  else {           /* single chain */
     Gscale = 1.0;
     Gmovx = 0;
  }
}

/********************************************/
/*resetview_cb:
 Togl Callback, reset all transformations
                                            */
/********************************************/
int
resetview_cb (struct Togl *togl, int argc, char *argv[])
{
  resetpara();

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* resetview_cb */

/********************************************/
/*update3DView_cb:
 Togl Callback, called on slider movement
 calculates a new polymer and redisplays      */
/********************************************/
int
update3DView_cb (struct Togl *togl, int argc, char *argv[])
{
  int value;

  if (argc > 3) {
     value = atof (argv[3]);

     if (strcmp (argv[2], "temperature") == 0) 
        arg.temperature = value;
     else if (strcmp (argv[2], "length") == 0) 
        arg.max_length = value;
     else if (strcmp (argv[2], "seed") == 0) 
        arg.seed = value;
     else if (strcmp (argv[2], "regionL") == 0) 
        arg.M = value;
     else if (strcmp (argv[2], "regionH") == 0) 
        arg.Y = value;
     else if (strcmp (argv[2], "regionW") == 0) 
        arg.Z = value;
     else if (strcmp (argv[2], "density") == 0) 
        arg.D = value;
     else if (strcmp (argv[2], "Lattice") == 0) 
        arg.lattice = value;
     else if (strcmp (argv[2], "Type") == 0) 
        arg.type = value;
     else
        /* fprintf(stderr, "No %s Type\n", argv[2]); */
        ;
  }
  
  polymer=1;
  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* update3DView */

/********************************************/
/*cmdCreatePolymer:
  a Tcl-command that interfaces to 
  Polymer-Lib                                */
/********************************************/
int
cmdCreatePolymer (ClientData clientData, Tcl_Interp * interp,
		int argc, char *argv[])
{
  polymer = 1;
  return TCL_OK;
}				/* cmdCreatePolymer */

/********************************************/
/*cmdDeletePolymer:
  a Tcl-command that interfaces to 
  Polymer-Lib                                */
/********************************************/
int
cmdDeletePolymer (ClientData clientData, Tcl_Interp * interp,
		int argc, char *argv[])
{
  polymer = 0;

  return TCL_OK;
}				/* cmdDeletePolymer */

/********************************/
/* setslider:
  sets the values from
  PLarg into Tcl-Vars          */
/********************************/
void
setslider (Tcl_Interp * interp, char *var, int val)
{
  char valstring[250], *getval, varname[255];
  double minslider = 0.0, maxslider = 0.0;

  sprintf (valstring, "%d", val);

  /* before setting we must check (and correct) the bounds */
  sprintf (varname, "%smin", var);
  if((getval = Tcl_GetVar (interp, varname, TCL_LEAVE_ERR_MSG)) == NULL)
    fprintf(stderr,"oops, cannot get variable %s\n", varname);
  else
    Tcl_GetDouble (interp, getval, &minslider);

  if (val < minslider)
    {
      /* We don't configure the slider here, but we set the
       * associated variable */
      if(Tcl_SetVar (interp, varname, valstring, TCL_LEAVE_ERR_MSG) == NULL)
	fprintf(stderr,"oops, cannot set variable %s\n", varname);
    }

  sprintf (varname, "%smax", var);
  if((getval = Tcl_GetVar (interp, varname, TCL_LEAVE_ERR_MSG)) == NULL)
    fprintf(stderr,"oops, cannot get variable %s\n", varname);
  else
    Tcl_GetDouble (interp, getval, &maxslider);

  if (val > maxslider)
    {
      if(Tcl_SetVar (interp, varname, valstring, TCL_LEAVE_ERR_MSG) == NULL)
	fprintf(stderr,"oops, cannot set variable %s\n", varname);
    }

    sprintf (valstring, "%d", val);

  /* Now, set the value */
  if(Tcl_SetVar (interp, var, valstring, TCL_LEAVE_ERR_MSG) == NULL)
    fprintf(stderr,"oops, cannot set variable %s\n", var);
  getval = Tcl_GetVar (interp, var, TCL_LEAVE_ERR_MSG);

  /*  fprintf(stderr,"Set value: %s , get value %s\n",valstring,getval);*/

  return;
} /* setslider */


/********************************************/
/*cmdReadCH2:
  a Tcl-command that loads a .ch2 
  (Datafile for Polymer-Lib)                 */
/********************************************/
int
cmdReadCH2 (ClientData clientData, Tcl_Interp * interp,
	    int argc, char *argv[])
{
  FILE *fileptr = NULL;
  int pass;

  if (argc != 3) {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"readCH2 ?filename?\"",
		     TCL_STATIC);
      return TCL_ERROR;
  }
  pass = atoi(argv[2]);

  if ((argv[1] != NULL) && ((fileptr = fopen (argv[1], "r")) == NULL)) {
      Tcl_SetResult (interp, "unable to open file", TCL_STATIC);
      return TCL_ERROR;
  }

  ReadInfile (&arg, fileptr);
  if (arg.type)
     Gscale = 0.3;
  else
     Gscale = 1.0;

  /* set Tcl-Vars */
    
  if (pass) { /* if pass = 0, return Lattice and Type */ 
     resetpara ();
     setslider (interp, "temperature", arg.temperature);
     setslider (interp, "seed", arg.seed);
     setslider (interp, "length", arg.max_length);
     setslider (interp, "regionL", arg.M);
     setslider (interp, "regionH", arg.Y);
     setslider (interp, "regionW", arg.Z);
     setslider (interp, "density", arg.D);
  }

  return TCL_OK;
}				/* cmdReadCH2 */


/********************************************/
/*cmdWriteCH2:
  a Tcl-command that writes a .ch2 
  (Datafile for Polymer-Lib)                 */
/********************************************/
int
cmdWriteCH2 (ClientData clientData, Tcl_Interp * interp,
	     int argc, char *argv[])
{
  FILE *fileptr = NULL;

  if (argc != 2)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"writeCH2 ?filename?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  if ((fileptr = fopen (argv[1], "w")) == NULL)
    {
      Tcl_SetResult (interp, "unable to open file", TCL_STATIC);
      return TCL_ERROR;
    }

  WriteRAWFile (arg, fileptr);

  fclose(fileptr);

  return TCL_OK;
}				/* cmdWriteCH2 */


/********************************************/
/*cmdWrite3D:
  a Tcl-command that writes a 3D Object 
  from the current polymer                    */
/********************************************/
int
cmdWrite3D (ClientData clientData, Tcl_Interp * interp,
	    int argc, char *argv[])
{
  FILE *fp = NULL;
  int i0, j0, k0, output, clip; 
  int side = 0, length = arg.max_length;
  int inRegion = 0;

  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"write3D type filename\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  if ((fp = fopen (argv[2], "w")) == NULL)
    {
      Tcl_SetResult (interp, "unable to open file", TCL_STATIC);
      return TCL_ERROR;
    }

   if (clip_flag)
      clip = 2;
   else
      clip = 1;

  /* get type of file */
  output = atoi (argv[1]);

  switch (output) {
    case 0:
        WriteRAWFile (arg, fp);
        break;
    case 1:
        if (arg.type) { /* amorphous region */
           srand(arg.seed);
           for (k0 = 0; k0 <= doubleside_flag; k0++)
           for (i0 = -arg.Y+1; i0 < arg.Y; i0++)
           for (j0 = -arg.Z+1; j0 < arg.Z; j0++) {

             if (randfloat() < arg.D*0.01) {
                arg.x_init = 0; arg.y_init = i0; arg.z_init = j0;
 
                if (arg.lattice)
                   make_new_chain_diamond (k0, clip, &side, &length);
                else
                   make_new_chain_cubic (k0, clip, &side, &length);
 
                if (arg.mode) {
		    fprintf(stderr, "3D amorphous region not implemented\n");
                    /* WriteWRL3DFile (inRegion, side, length, fp); */
		}
                else {
                    WriteWRLFile (inRegion, side, length, fp);
                    if (!inRegion)
		       inRegion = 1;
		    }
              }
           }
        }
        else {
            if (arg.mode)
                WriteWRL3DFile (1, side, length, fp);
            else
                WriteWRLFile (1, side, length, fp);
        }
        break;
    case 2:
        if (arg.type) { /* amorphous region */
           srand(arg.seed);
           for (k0 = 0; k0 <= doubleside_flag; k0++)
           for (i0 = -arg.Y+1; i0 < arg.Y; i0++)
           for (j0 = -arg.Z+1; j0 < arg.Z; j0++) {
 
             if (randfloat() < arg.D*0.01) {
                arg.x_init = 0; arg.y_init = i0; arg.z_init = j0;
 
                if (arg.lattice)
                   make_new_chain_diamond (k0, clip, &side, &length);
                else
                   make_new_chain_cubic (k0, clip, &side, &length);
 
                if (arg.mode) {
                    fprintf(stderr, 
		      "Wavefront file of 3D polymer chain is not implemented\n");
                    /* WriteWavefrontFile (inRegion, side, length, fp); */
                }
                else {
                    WriteWavefrontFile (inRegion, side, length, fp);
                    if (!inRegion)
                       inRegion = 1;
                    }
              }
           }
        }
        else {
            if (arg.mode) {
                fprintf(stderr, "3D amorphous region not implemented\n");
                /* WriteWavefrontFile (1, side, length, fp); */
            }
            else
                WriteWavefrontFile (-1, side, length, fp);
        }
        break;
    }

  fclose(fp);

  return TCL_OK;
}				/* cmdWrite3D */


int
cmdDumpScreen (ClientData clientData, Tcl_Interp * interp,
                int argc, char *argv[])
{
  extern StartDump(char *, int, int, int);
  extern EndDump(void);
  int orient;
 
  orient = 0;

  StartDump(argv[1], orient, width, height);
  EndDump();
}
 
 
Tcl_Interp * TheInterp;
/********************************************/
/*Tcl_AppInit:                              */
/********************************************/
int
Tcl_AppInit (Tcl_Interp * interp)
{
  if (Tcl_Init (interp) == TCL_ERROR)
    return TCL_ERROR;

  if (Tk_Init (interp) == TCL_ERROR)
    return TCL_ERROR;

  if (Togl_Init (interp) == TCL_ERROR)
    return TCL_ERROR;

  TheInterp = interp;

  Togl_CreateFunc (create_cb);
  Togl_DisplayFunc (display_cb);
  Togl_ReshapeFunc (reshape_cb);

  Togl_CreateCommand ("setXrot", setXrot_cb);
  Togl_CreateCommand ("setYrot", setYrot_cb);
  Togl_CreateCommand ("setZrot", setZrot_cb);
  Togl_CreateCommand ("zoomIn", zoomin_cb);
  Togl_CreateCommand ("zoomOut", zoomout_cb);
  Togl_CreateCommand ("moveX", movex_cb);
  Togl_CreateCommand ("moveY", movey_cb);
  Togl_CreateCommand ("moveZ", movez_cb);
  Togl_CreateCommand ("update3DView", update3DView_cb);
  Togl_CreateCommand ("setMode", setmode_cb);
  Togl_CreateCommand ("ColorRevert", colorrevert_cb);
  Togl_CreateCommand ("setClip", setclip_cb);
  Togl_CreateCommand ("teachingAid", teaching_aid_cb);
  Togl_CreateCommand ("setShow", setshow_cb);
  Togl_CreateCommand ("DoubleSide", setdoubleside_cb);
  Togl_CreateCommand ("drawSys", drawsys_cb);
  Togl_CreateCommand ("resetView", resetview_cb);
  Togl_CreateCommand ("CH2", show_H_cb);

  Tcl_CreateCommand (interp, "createPolymer", cmdCreatePolymer,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "deletePolymer", cmdDeletePolymer,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "readCH2", cmdReadCH2,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "writeCH2", cmdWriteCH2,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "write3D", cmdWrite3D,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "dumpScreen", cmdDumpScreen,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_LinkVar (interp, "cvch2", (char *) &arg.ch2, TCL_LINK_INT); 
  Tcl_LinkVar (interp, "Lattice", (char *) &arg.lattice, TCL_LINK_INT); 
  Tcl_LinkVar (interp, "Type", (char *) &arg.type, TCL_LINK_INT); 
  Tcl_LinkVar (interp, "seed", (char *) &arg.seed, TCL_LINK_INT); 
  Tcl_LinkVar (interp, "temperature", (char *) &arg.temperature, TCL_LINK_INT);  
  Tcl_LinkVar (interp, "max_length", (char *) &arg.max_length, TCL_LINK_INT); 
  Tcl_LinkVar (interp, "regionL", (char *) &arg.M, TCL_LINK_INT); 
  Tcl_LinkVar (interp, "regionH", (char *) &arg.Y, TCL_LINK_INT); 
  Tcl_LinkVar (interp, "regionW", (char *) &arg.Z, TCL_LINK_INT); 
  Tcl_LinkVar (interp, "density", (char *) &arg.D, TCL_LINK_INT); 

  return TCL_OK;
} /* Tcl_AppInit */


/********************************************/
/*main:                                     */
/********************************************/
int
main (int argc, char **argv)
{
  Tk_Main (argc, argv, Tcl_AppInit);

  return 0;
} /* main */

