# pragma ident "@(#)lab.tcl	1.1 06/24/97 Polymer
# PolymerLibV1.0 - the PolymerGenerator 
#
# Copyright (C) 1997 S. F. Bellenot (bellenot@math.fsu.edu)
#                  & Z.-H. Duan (zduan@math.fsu.edu)
#
# This software is shareware!
# Read the file "License" for further information.

#
# The Polymer Laboratory, a Tcl/Tk - OpenGL GUI for PolymerLib
# Tcl/Tk part
#

# Vars, lots of ;)
set cvupdate_while_sliding 1
set cvdrawSys 1
set cvClip 1
set cvShow 1
set cvDoubleside 0
set cvta 0
set cvColor 0
set cvch2 0
set 3dview_open "no"
set oldXpos 0
set oldYpos 0
set oldZpos 0
set Mode 0
set TA 0

set filename ""

# the entries are bound to dummy, when focused
set dummy 0
set dummyval 0
set safedummy 0

# the sliders are bound to dummyval, when loading a file
set dummyval2 0

# all sliders, we derive varnames & widgetnames from this list
set sliders { max_length temperature seed regionL regionH regionW density }

# initial values 
#(max_length temperature seed regionL regionH regionW density) for all sliders
set sinit { 200 1000 105 20 10 10 10 }

# initial minimum values for all sliders
set sinitmin { 0 0 0 0 0 0 0 }

# initial maximum values for all sliders
set sinitmax { 1000 1000 10000 100 100 100 100 }

set index 0

# generate global variables
foreach slider $sliders {

    set maxname  $slider
    set minname  $slider
    append minname "min"
    append maxname "max"

    set $slider [lindex $sinit $index]
    set $minname [lindex $sinitmin $index]
    set $maxname [lindex $sinitmax $index]

    incr index
}

##############################
##############################
# procedures

proc demo {} {
   global max_length;

   while {$max_length > 4 } {
      set max_length [expr $max_length - 5];
      demo; update idletasks;
   }
}

##############################
# helpTut:
# open tutorial-window
#
proc helpTut {} {
    set w .hlptut
    catch {destroy $w}
    toplevel $w
    wm title $w "Polymer Laboratory Help: Tutorial"
    wm iconname $w "PolymerLabHelp"
    wm group $w .

    frame $w.ftext
    frame $w.fbutton
    pack $w.ftext -in $w -side top -expand yes -fill both
    pack $w.fbutton -in $w -side bottom

    text $w.ftext.text -yscrollcommand "$w.ftext.sbar set" \
 	  -setgrid 1 -height 25 -width 70
    scrollbar $w.ftext.sbar -takefocus 0 -command "$w.ftext.text yview"
    pack $w.ftext.text -in $w.ftext -side left -fill both -expand yes
    pack $w.ftext.sbar -in $w.ftext -side right -fill y
    $w.ftext.text insert end \
{ 

Polymer Laboratory Tutorial
===========================

This is a small introduction to the Polymer-Laboratory.

You may keep this window open while experimenting 
with the Polymer Laboratory.

Our goal is to provide a tool to visualize the spaghetti
looking polymers. 
 
There are several parameters you can adjust in the main menu:
 
temperature --- see temperature effect on the configuration of
                polyethylene;
max_length  --- get a long or a short polyethylene;
seed        --- see another polyethylene;
regionL     --- adjust thickness of liquid like region;
regionH/
regionW     --- specify the size of the liquid like region;
density     --- change the density of polymers in the liquid 
		like region;  

(note: you may be not see all of them at certain mode.)
 
Play on them!
 
The polymer is now complete. Switch to the 3D-View. You may
 
1) shift the polymer by moving the mouse with the first button down;
2) rotate the polymer by moving the mouse with the second button down;
3) Zoom in/out the polymer by moving the mouse with the third button down;
4) Zoom in the polymer by pressing the key '+';
5) Zoom out the polymer by pressing the key '-';
6) Adjust the view again by pressing the key 'Down';
7) Adjust the view again by pressing the key 'Up';
8) Adjust the view again by pressing the key 'Right';
9) Adjust the view again by pressing the key 'Left'.
10) Reset the view by pressing the key 'r'.
 
Want save you work?
You may use the keyboard shortcut <Ctrl+s>.  A file requester pops up.
 
Now there are six pull-down menus with the main menu. You may use
 
1) File menu to save, load a polymer data file and quit from
   the Polymer Laboratory;
2) Export menu to save the polymer in the 3D-View window as
   .tiff, .wavefront, .wrl files.
3) Polymer menu to choose a single polymer chain or the liquid like
   region.
4) Model menu to switch from cubic lattice model to diamond lattice
   model, or other way around.
5) Rendering menu to pick different Rendering mode.
6) View menu to set some controls.

If you want to say Good Bye now, Press 'Control q'.

Debug button may help you set parameters manually or, of course, debug.
}

    $w.ftext.text configure -state disabled

    button $w.fbutton.b -text "Dismiss" -command "destroy $w"
    pack $w.fbutton.b -in $w.fbutton
}

##############################
# helpLab:
# opens PolymerLab-help-window
#
proc helpLab {} {
    set w .hlptut
    catch {destroy $w}
    toplevel $w
    wm title $w "Polymer Laboratory Help: Polymer Laboratory"
    wm iconname $w "PolymerLabHelp"
    wm group $w .

    frame $w.ftext
    frame $w.fbutton

    pack $w.ftext -in $w -side top -expand yes -fill both
    pack $w.fbutton -in $w -side bottom

    text $w.ftext.text -yscrollcommand "$w.ftext.sbar set" \
	-setgrid 1 -height 25 -width 70
    scrollbar $w.ftext.sbar -takefocus 0 -command "$w.ftext.text yview"
    pack $w.ftext.text -in $w.ftext -side left -fill both -expand yes
    pack $w.ftext.sbar -in $w.ftext -side right -fill y

    $w.ftext.text insert end \
{
Please refer to the real manual located in the "doc/" subdirectory of
the distribution for the documentation of the Polymer Laboratory.
}

    $w.ftext.text configure -state disabled

    button $w.fbutton.b -text "Dismiss" -command "destroy $w"
    pack $w.fbutton.b -in $w.fbutton
}

##############################
# helpAbout:
# opens about-window
#
proc helpAbout {} {
    set w .hlpabout
    catch {destroy $w}
    toplevel $w
    wm title $w "Polymer Laboratory Help: About"
    wm iconname $w "PolymerLabHelp"
    wm group $w .
    wm resizable $w no no

    frame $w.ftext
    frame $w.fbutton
    pack $w.ftext $w.fbutton -in $w -side top

    text $w.ftext.text -yscrollcommand "$w.ftext.sbar set" \
	-setgrid 1 -height 14 -width 50
    scrollbar $w.ftext.sbar -takefocus 0 -command "$w.ftext.text yview"
    pack $w.ftext.text -in $w.ftext -side left -fill both -expand yes
    pack $w.ftext.sbar -in $w.ftext -side right -fill y

    $w.ftext.text insert end \
{ 
Polymer Laboratory

A Tcl/Tk - OpenGL GUI for
PolymerLibV1.0 the PolymerGenerator
Copyright (c) 1997 by S. F. Bellenot & Z.-H. Duan
( bellenot@math.fsu.edu & zduan@math.fsu.edu )
All rights reserved.
This software is shareware!
}
    $w.ftext.text tag add "tag1" 1.0 end
    $w.ftext.text tag add "name" 2.0 2.end
    $w.ftext.text tag configure tag1 -justify center
    $w.ftext.text tag configure name -underline yes

    $w.ftext.text configure -state disabled

    button $w.fbutton.b -text "Dismiss" -command "destroy $w"
    pack $w.fbutton.b -in $w.fbutton
}

##############################
# rotate3DView:
# rotate 3D-View with mouse
#
proc rotate3DView { width height x y } {
    set w .3dwin

    $w.f3d.togl setXrot [expr 360.0 * $y / $height]
    $w.f3d.togl setYrot [expr 360.0 * ($width - $x) / $width]
}

##############################
# open3D:
# open 3D-View window
#
proc open3D {} {
    global 3dview_open
    global oldXpos
    global oldYpos
    global oldZpos

    set w .3dwin
    catch {destroy $w}
    toplevel $w
    wm title $w "Polymer Laboratory 3D-View"
    wm iconname $w "View"
    wm group $w .

    frame $w.f3d
    togl $w.f3d.togl -rgba true\
	    -double true -depth true -ident PolymerView

    # Bindings for the 3D-View
    bind $w.f3d.togl <B1-Motion> {
	if {%x > $oldXpos} {
	   .3dwin.f3d.togl moveX 1
	} elseif {%x < $oldXpos} { 
           .3dwin.f3d.togl moveX -1
	}
	if {%y > $oldYpos} {
	   .3dwin.f3d.togl moveY -1
	} elseif {%y < $oldYpos} { 
           .3dwin.f3d.togl moveY 1
	}
        set oldXpos %x
        set oldYpos %y
    }
    bind $w.f3d.togl <B2-Motion> {
	rotate3DView [lindex [%W config -width] 4] \
		     [lindex [%W config -height] 4] \
		     %x %y
    }
    bind $w.f3d.togl <B3-Motion> {
        if {%y > $oldYpos} {
          .3dwin.f3d.togl zoomIn
        } elseif {%y < $oldYpos} {
          .3dwin.f3d.togl zoomOut
	}
        set oldYpos %y
    }
    bind $w <KeyPress-plus> {
	.3dwin.f3d.togl zoomIn
    }
    bind $w <KeyPress-KP_Add> {
	.3dwin.f3d.togl zoomIn
    }
    bind $w <KeyPress-minus> {
	.3dwin.f3d.togl zoomOut
    }
    bind $w <KeyPress-KP_Subtract> {
	.3dwin.f3d.togl zoomOut
    }
    bind $w <KeyPress-Up> {
	.3dwin.f3d.togl moveZ  1
    }
    bind $w <KeyPress-Down> {
	.3dwin.f3d.togl moveZ -1
    }
    bind $w <KeyPress-Left> {
	.3dwin.f3d.togl setZrot -10
    }
    bind $w <KeyPress-Right> {
	.3dwin.f3d.togl setZrot 10
    }
    bind $w <KeyPress-r> {
	.3dwin.f3d.togl resetView
    }
    pack $w.f3d -in $w -fill both -expand yes
    pack $w.f3d.togl -in $w.f3d -side left -fill both -expand yes

    set 3dview_open "yes"
}

##############################
# setSHOW:
# update 3D-View
#
proc setSHOW {value} {
    global cvupdate_while_sliding
    global 3dview_open
 
    if { $3dview_open == "yes" } {
        if { $cvupdate_while_sliding == 1 } {
            .3dwin.f3d.togl setShow $value
        }
    }
}
 
##############################
# setMODE:
# update 3D-View
#
proc setMODE {value} {
    global cvupdate_while_sliding
    global 3dview_open

    if { $3dview_open == "yes" } {
	if { $cvupdate_while_sliding == 1 } {
	    .3dwin.f3d.togl setMode $value
	}
    }
}

##############################
# updateView:
# update 3D-View
#
proc updateView {name value} {
    global cvupdate_while_sliding
    global 3dview_open

    if { $3dview_open == "yes" } {
	if { $cvupdate_while_sliding == 1 } {
	    .3dwin.f3d.togl update3DView $name $value
	}
    }
}

##############################
# setminmax:
# set a sliders from, to and resolution values
#
proc setminmax {min max paramname} {

    set slfw .fmid.fsliders.canv.slidecon.$paramname

    global $min $max $paramname

    set $min [.calwindow.emin get]
    set $max [.calwindow.emax get]

    $slfw.lmin configure -text [.calwindow.emin get]
    $slfw.lmax configure -text [.calwindow.emax get]

    $slfw.s configure -from [.calwindow.emin get]
    $slfw.s configure -to [.calwindow.emax get]
    $slfw.s configure -resolution [.calwindow.step get]
}


##############################
# calibrateSlider:
# request new from, to and resolution values of a slider
#
proc calibrateSlider {paramname} {

    set slfw .fmid.fsliders.canv.slidecon.$paramname

    # create a new window
    set w .calwindow
    catch {destroy $w}
    toplevel $w
    wm title $w "Calibrate Slider: $paramname"
    wm group $w .
    wm iconname $w "Cal."
    wm resizable $w no no

    # fetch values
    set min $paramname 
    set max $paramname 
    append min "min"
    append max "max"
    global $min $max $paramname

    eval set rmin \$$min
    eval set rmax \$$max
    frame $w.entries
    frame $w.entriesr
    frame $w.entriesl
    frame $w.buttons

    pack $w.entries -in $w -side top 
    pack $w.entriesr -in $w.entries -side right
    pack $w.entriesl -in $w.entries -side left
    pack $w.buttons -in $w -side bottom 

    # create 3 labeled entries
    label $w.lmin -text "$min:" -width 16
    label $w.lmax -text "$max:" -width 16
    label $w.lstep -text "Stepsize:" -width 16

    entry $w.emin -width 16
    $w.emin insert @0 $rmin

    entry $w.emax -width 16 
    $w.emax insert @0 $rmax

    entry $w.step -width 16
    set step [$slfw.s cget -resolution]
    $w.step insert @0 $step

    pack $w.lmin -in $w.entriesl -side top 
    pack $w.lmax -in $w.entriesl -side top 
    pack $w.lstep -in $w.entriesl -side top

    pack $w.emin -in $w.entriesr -side top 
    pack $w.emax -in $w.entriesr -side top 
    pack $w.step -in $w.entriesr -side top 

    # create ok & cancel button
    button $w.buttons.ok -text "  Ok!  " \
	-command "setminmax $min $max $paramname; destroy $w"

    button $w.buttons.cancel -text "Cancel!" -command "destroy $w"

    pack $w.buttons.ok $w.buttons.cancel -in $w.buttons -side left\
	-padx 4m -pady 2m 
    grab set $w
}


##############################
# createSlider:
# create a slider and associated entry & labels
#
proc createSlider { name slname } {
    global Lattice
    global Type

    set min $name 
    set max $name 
    append min "min"
    append max "max"
    global $min $max $slname $name

    # Don't you think using strings as varnames is a funny thing? ;)

    eval set rmin \$$min
    eval set rmax \$$max
    eval set val \$$name
    frame $slname.$name

    # parameter name
    if { $name == "density" } {
        label $slname.$name.l -text "density(%)" -width 12
    } else {
        label $slname.$name.l -text "$name " -width 12
    }

    # Entry showing the real value
    entry $slname.$name.e -textvariable $name -width 12

    bind $slname.$name.e <FocusIn> {
        set dummy [%W cget -textvariable]
        set dummyval [%W get]
        set safedummy $dummyval
        %W configure -textvariable dummyval 
    }
    bind $slname.$name.e <FocusOut> {
        eval set newval \$$dummy
	if { [catch {set $dummy $dummyval}] == 1} { set $dummy $safedummy }
	%W configure -textvariable $dummy
	if { $newval != $dummyval } { updateView $dummy [%W get] }
    }
    # the slider itself 
    scale $slname.$name.s  -orient horizontal -showvalue no\
        -length 5c -variable $name -from $rmin -to $rmax\
	-command "updateView $name"

    bind $slname.$name.s <Button-1> { focus %W }

    #calc resolution, always a 1000th of the next higher full dezimal
    set a [expr ($rmax - $rmin)*10.0]
    set b 1.0
    while {$a > 1.0} {
        set a [expr $a/10.0]
        set b [expr $b*10.0]
    }
    #$slname.$name.s configure -resolution [expr ($b/10000.0)]

    $slname.$name.s configure -from $rmin
    $slname.$name.s configure -to $rmax

    #now that the sliders resolution is correct set, we may set a value
    $slname.$name.s set $val

    # Label showing the minval of the slider
    label $slname.$name.lmin -text $rmin -width 5

    # Label showing the maxval of the slider
    label $slname.$name.lmax -text $rmax -width 5

    # Calibrate button
    button $slname.$name.b -text "Cal." -command "calibrateSlider $name"\
    	-takefocus 0

    pack $slname.$name.l $slname.$name.e \
	-in $slname.$name -side left -padx 2m
    pack $slname.$name.lmin $slname.$name.s\
	$slname.$name.lmax\
	-in $slname.$name -side left
    pack $slname.$name.b -side right -padx 2m

    if {$name == "max_length"} {
        label $slname.laal -text "Polymer Parameters:"
        pack $slname.laal -in $slname -pady 2m
    }
    if {$Lattice == 0 && $name == "temperature"} {
    } elseif {$Type == 0} {
         if {($name == "regionL") ||
             ($name == "regionH") ||
             ($name == "regionW") ||
	     ($name == "density")} {
         } else {
	      pack $slname.$name -in $slname -pady 1m
         }
    } else {
           pack $slname.$name -in $slname -pady 1m
    }
}


############################
# gotoDir:
#
proc gotoDir {dir} {
    global path
    set w .reqwin
    catch {cd $dir}
    set path [pwd]
    $w.upper.flist delete 0 end
    $w.upper.flist insert end ".."

    foreach i [lsort [glob -nocomplain */]] {
	$w.upper.flist insert end $i
    }
    foreach i [lsort [glob -nocomplain *]] {
	if {[file isdirectory $i] == 0} { $w.upper.flist insert end $i }
    }
    $w.upper.flist selection set 0
}


############################
# processFile:
#
proc processFile {file} {
    global filename
    set w .reqwin

    if { $file != "" } {
        if {[file exists $file]} {
    	    if {[file isdirectory $file] == 1} {
	        gotoDir $file
	    } else {
	        if {[file type $file] == "file"} {
	    	    set filename $file
		    destroy $w
	        }
	    }

        } else {
    	    set filename $file
	    destroy $w
        }
    }
}


##############################
# requestFilename:  
# a little funny filerequester, ASL-like
# Amiga-Users will know ;)
#
proc requestFileName { name } {
    global filename
    set path [pwd]
    set file $filename

    set w .reqwin
    catch {destroy $w}
    toplevel $w
    wm title $w "$name"
    wm resizable $w no yes
    wm iconname $w "SelFile"
    wm group $w .


    frame $w.upper
    listbox $w.upper.flist -relief sunken -selectmode browse\
	-yscrollcommand "$w.upper.sbar set" -exportselection yes

    scrollbar $w.upper.sbar -takefocus 0 -command "$w.upper.flist yview"

    pack $w.upper.flist -in $w.upper -side left -fill both -expand yes
    pack $w.upper.sbar -in $w.upper -side right -fill y

    pack $w.upper -in $w -side top -expand yes -fill both

    $w.upper.flist insert end ".."

    foreach i [lsort [glob -nocomplain */]] {
     $w.upper.flist insert end $i
    }

    foreach i [lsort [glob -nocomplain *]] {
        if {[file isdirectory $i] == 0} { $w.upper.flist insert end $i }
    }
    $w.upper.flist selection set 0

    # keybindings for our listbox
    bind $w.upper.flist <Double-Button-1> {
        processFile [%W get [%W nearest %y]]}
    bind $w.upper.flist <KeyPress-Return> {
        processFile [%W get [%W curselection]]}

    bind $w.upper.flist <Button-1> {
        if {[file isdirectory [%W get [%W nearest %y]]] == 0 } {
    	    set file [%W get [%W nearest %y]]
        }
    }
    bind $w.upper.flist <KeyPress-Up> {
        if {[file isdirectory [%W get [expr [%W curselection] - 1]]] == 0} {
	    set file [%W get [expr [%W curselection] - 1]]
        }
    }
    bind $w.upper.flist <KeyPress-Down> {
        if {[file isdirectory [%W get [expr [%W curselection] + 1]]] == 0} {
	    set file [%W get [expr [%W curselection] + 1]]
        }
    }
    bind $w.upper.flist <Button-3> { gotoDir ".." }

    frame $w.entry1
    pack $w.entry1 -in $w  -fill x
    frame $w.entry2
    pack $w.entry2 -in $w  -fill x
    frame $w.buttons 
    pack $w.buttons -in $w -side bottom -fill x

    # create 2 labeled entries
    label $w.entry1.lpath -text "Path:"
    label $w.entry2.lfile -text "File:"

    entry $w.entry1.epath -textvariable path
    $w.entry1.epath delete 0 end
    $w.entry1.epath insert @0 $path

    entry $w.entry2.efile -textvariable file
    $w.entry2.efile delete 0 end
    $w.entry2.efile insert @0 $file

    pack $w.entry1.lpath -in $w.entry1 -side left -fill x -padx 2m
    pack $w.entry1.epath -in $w.entry1 -side left -expand yes -fill x -padx 2m
    pack $w.entry2.lfile -in $w.entry2 -side left -fill x -padx 2m
    pack $w.entry2.efile -in $w.entry2 -side left -expand yes -fill x -padx 2m

    # keybindings for the entries
    bind $w.entry1.epath <KeyPress-Return> { gotoDir $path }
    bind $w.entry1.epath <FocusOut> { gotoDir $path }

    bind $w.entry2.efile <KeyPress-Return> { processFile $file }

    # create ok, parent & cancel button
    button $w.buttons.ok -text "  Ok!  " -command {processFile $file}
    button $w.buttons.parent -text "Parent" -command {gotoDir ".."}
    button $w.buttons.cancel -text "Cancel!" -command { 
        set filename ""; destroy .reqwin
    }

    pack $w.buttons.ok $w.buttons.parent $w.buttons.cancel\
	-in $w.buttons -expand yes -side left\
	-padx 4m -pady 2m  -fill x

    focus $w.upper.flist
    grab set $w
}

##############################
# readFile:  
# 
proc readFile { } {
   global Lattice
   global Type
   global sliders
   global slfw
   global filename
   global dummyval2

   set filename ""

   requestFileName "Select file to load:"
   tkwait window .reqwin

   if {[file exists $filename]} {

      if { $filename != "" } {

        readCH2 $filename 0
	createCanvas $Lattice $Type
 	pack forget .fmid.fsliders	

	#disconnect sliders from variables
	foreach slider $sliders {  
           $slfw.$slider.s configure -variable dummyval2
           $slfw.$slider.e configure -textvariable dummyval2
	}

	foreach slider $sliders {    
	    set min $slider 
	    set max $slider 
	    append min "min"
	    append max "max"
	    global $min $max $slider
	}
        wm title . "Polymer Laboratory - $filename"
	readCH2 $filename 1

	foreach slider $sliders {    
	    set min $slider 
	    set max $slider 
	    append min "min"
	    append max "max"
	    eval set rmin \$$min
	    eval set rmax \$$max
	    eval set val \$$slider
	    
            $slfw.$slider.s configure -from $rmin -to $rmax
            $slfw.$slider.s set $val
	    $slfw.$slider.lmin configure -text $rmin
	    $slfw.$slider.lmax configure -text $rmax

	    $slfw.$slider.s configure -variable $slider
	    $slfw.$slider.e configure -textvariable $slider
	}
	pack .fmid.fsliders -side left -in .fmid -fill y -expand yes

	#reset the radio button 

	createRenderingbotton;
        createViewbotton;
	if { $Type } {
     	   .fmenuBar.polymer.m invoke 2 
        } else {
	   .fmenuBar.polymer.m invoke 1
        }
	if { $Lattice } {
     	   .fmenuBar.model.m invoke 2 
        } else {
	   .fmenuBar.model.m invoke 1
        }
      }
   }
}

##############################
# writeFile:
# 
proc writeFile {filetype extension} {
    global filename

    if { $filename == "" } {
        set filename "unnamed"
        append filename  $extension
    } else {
        set filename [file rootname $filename]
        append filename $extension
    }
    requestFileName "Select file to save to:"
    tkwait window .reqwin

    $filetype $filename 
}

##############################
# write3DObject:
# save 3D Object
#
proc teachingAid {value } {
    global cvupdate_while_sliding
    global 3dview_open
 
    if { $3dview_open == "yes" } {
        if { $cvupdate_while_sliding == 1 } {
            .3dwin.f3d.togl teachingAid $value
        }
    }

}

##############################
# write3DObject:
# save 3D Object 
#
proc write3DObject { type extension } {
    global filename

    if { $filename == "" } { 
        set filename "unnamed"
        append filename  $extension
    } else {
        set filename [file rootname $filename]
        append filename $extension
    }

    requestFileName "Select file to save to:"

    tkwait window .reqwin

    if { $filename != "" } {
        #    puts stdout "Writing to file $filename !"
        write3D $type $filename
    }
}

##############################
# a generic interactor for tcl stolen from vtk
#
set dbInteractBold "-background #43ce80 -relief raised -borderwidth 1"
set dbInteractNormal "-background #c0c0c0 -relief flat"
set dbInteractTagcount 1;

proc dbInteract {} {

    proc dodb {s w} {
	global dbInteractBold dbInteractNormal dbInteractTagcount;
	set tag [append tagnum $dbInteractTagcount];
	incr dbInteractTagcount 1;
	.dbInteract.text configure -state normal
	.dbInteract.text insert end $s $tag
	eval .dbInteract.text tag configure $tag $dbInteractNormal
	.dbInteract.text tag bind $tag <Any-Enter> \
	    ".dbInteract.text tag configure $tag $dbInteractBold"
	.dbInteract.text tag bind $tag <Any-Leave> \
	    ".dbInteract.text tag configure $tag $dbInteractNormal"
	.dbInteract.text tag bind $tag <1> "dodb [list $s] .dbInteract";
	.dbInteract.text insert end \n; 
	.dbInteract.text insert end [uplevel 1 $s]; 
	.dbInteract.text insert end \n\n; 
	.dbInteract.text configure -state disabled
	.dbInteract.text yview end
    }

    catch {destroy .dbInteract}
    toplevel .dbInteract
    wm title .dbInteract "debug Interactor"
    wm iconname .dbInteract "debug"
    
    frame .dbInteract.buttons
    pack  .dbInteract.buttons -side bottom -fill x -pady 2m
    button .dbInteract.buttons.dismiss -text Dismiss \
	-command "wm withdraw .dbInteract"
    pack .dbInteract.buttons.dismiss -side left -expand 1

    button .dbInteract.buttons.clear -text Clear \
	-command "cleardbInteract"
    pack .dbInteract.buttons.clear -side left -expand 1
    
    frame .dbInteract.file
    label .dbInteract.file.label -text "Command:" -width 10 -anchor w
    entry .dbInteract.file.entry -width 40 
    bind .dbInteract.file.entry <Return> {
	dodb [%W get] .dbInteract; %W delete 0 end;\
	updateView NULL 0} 
    pack .dbInteract.file.label -side left
    pack .dbInteract.file.entry -side left -expand 1 -fill x
    
    text .dbInteract.text -yscrollcommand ".dbInteract.scroll set" \
	-setgrid true -width 60 -height 8 -wrap word
    scrollbar .dbInteract.scroll -command ".dbInteract.text yview"
    pack .dbInteract.scroll -side right -fill y
    pack .dbInteract.text -side bottom -expand 1 -fill both
    pack .dbInteract.file -pady 3m -padx 2m -side bottom -fill x 
    
    .dbInteract.text configure -state disabled
    wm withdraw .dbInteract
}

##############################
#
proc cleardbInteract {} {
	.dbInteract.text configure -state normal
	.dbInteract.text delete 0.0 end;
	#puts "clear db Interact"
	.dbInteract.text configure -state disabled
}

dbInteract;

##############################
# create slider-canvas
#
proc createCanvas {Lattice Type} {
   global sliders
   global slfw

   catch {destroy .fmid.fsliders}
   frame .fmid.fsliders 
   pack .fmid.fsliders -side left -in .fmid -fill y -expand yes

   canvas .fmid.fsliders.canv -yscrollcommand {.fmid.fsliders.scrollb set}\
    	-width 16c -height 6c\
 	-scrollregion {0c 0c 5c 25c}\
	-relief sunken -borderwidth 2 

   scrollbar .fmid.fsliders.scrollb -orient vertical\
	-command {.fmid.fsliders.canv yview}
   pack .fmid.fsliders.scrollb -in .fmid.fsliders -side right -fill y

   pack .fmid.fsliders.canv -in .fmid.fsliders -side left -fill both\
	-expand yes

   set slfw .fmid.fsliders.canv.slidecon
   catch {destroy $slfw}
   frame $slfw

   # now create all sliders + associated buttons & entrys
   foreach slider $sliders { 
       createSlider $slider $slfw 
   }
   .fmid.fsliders.canv create window 0c 0.3c -window $slfw -anchor nw\
   -tags item

}

########################################################
# View-menu
#
proc createViewbotton { } {
    global Mode
    global Lattice
    global Type

    catch {destroy .fmenuBar.view}

    menubutton .fmenuBar.view -text "View" -menu .fmenuBar.view.m\
	-underline 0

    menu .fmenuBar.view.m

    if { $Type == 1 } {
       .fmenuBar.view.m add radio -label "Show ALL" \
         -variable cvShow -command {setSHOW 1}

       .fmenuBar.view.m invoke 1

       .fmenuBar.view.m add radio -label "Loop Only" \
         -variable cvShow -command {setSHOW 2}
       .fmenuBar.view.m add radio -label "Tie Only" \
         -variable cvShow -command {setSHOW 3}
       .fmenuBar.view.m add radio -label "Free-end Only" \
         -variable cvShow -command {setSHOW 4}

       .fmenuBar.view.m add separator
    } elseif {$Lattice == 1} {
       .fmenuBar.view.m add check -label "Show Hydrogens too" \
	   -variable cvch2 \
	   -command "catch {.3dwin.f3d.togl CH2}"

#       .fmenuBar.view.m invoke 1
       .fmenuBar.view.m add separator
    }

    .fmenuBar.view.m add check -label "AutoUpdate"\
	-variable cvupdate_while_sliding\
        -command { if {$cvupdate_while_sliding == 1 } createPolymer }

    .fmenuBar.view.m add checkbutton -label "DrawCoordSys" \
	-variable cvdrawSys -command "catch {.3dwin.f3d.togl drawSys}"

###### amorphous region ######
    if { $Type == 1 } {  
       .fmenuBar.view.m add check -label "SetClip" \
   	 -variable cvClip -command "catch {.3dwin.f3d.togl setClip}"
       .fmenuBar.view.m add check -label "DoubleSide" \
   	 -variable cvDoubleside -command "catch {.3dwin.f3d.togl DoubleSide}"
    }
    if {$Mode == 0} {
       .fmenuBar.view.m add check -label "ColorRevert" \
         -variable cvColor -command "catch {.3dwin.f3d.togl ColorRevert}"
    }

    .fmenuBar.view.m add separator

    .fmenuBar.view.m add command -label "ResetView"\
	-command "catch {.3dwin.f3d.togl resetView}"

    .fmenuBar.view.m add command -label "ClearScreen"\
	-command { deletePolymer; \
		   catch {.3dwin.f3d.togl resetView} }

    pack .fmenuBar.view -in .fmenuBar -side left
}

########################################################
# Rendering-menu
#
proc createRenderingbotton { } {
    global Mode
    global Type

    catch {destroy .fmenuBar.pref}

    menubutton .fmenuBar.pref -text "Render" -menu .fmenuBar.pref.m\
 	-underline 0

    if { $Type == 0 } {
        menu .fmenuBar.pref.m

        .fmenuBar.pref.m add radio -label "Line" -variable MODE\
    	    -command {setMODE 0; set Mode 0} 
        .fmenuBar.pref.m invoke 1

        .fmenuBar.pref.m add radio -label "Wireframe" -variable MODE\
	    -command {setMODE 1; set Mode 1}

        .fmenuBar.pref.m add radio -label "FlatShaded" -variable MODE\
	    -command {setMODE 2; set Mode 2}

        .fmenuBar.pref.m add radio -label "SmoothShaded" -variable MODE\
	    -command {setMODE 3; set Mode 3}
    } else {
        menu .fmenuBar.pref.m

        .fmenuBar.pref.m add radio -label "Line" -variable MODE\
    	    -command {setMODE 0; set Mode 0} 
        .fmenuBar.pref.m invoke 1

    }

    pack .fmenuBar.pref -in .fmenuBar -side left
}

#####################################################
#####################################################
# main
wm title . "Polymer Laboratory"
wm iconname . "PolymerLab"
wm minsize . 1 1
wm geometry . 620x400
wm resizable . yes yes

# create frames
frame .fmenuBar -relief raise 
pack .fmenuBar -side top -fill x
frame .fbuttons 
pack .fbuttons -side bottom -fill x
frame .fmid
pack .fmid -fill both -expand yes


set Lattice 1
set Type 1

createCanvas $Lattice $Type


# create menu-bar---------------------------------------------------

# Quit-menu
#button .fmenuBar.quit -text "Quit" -command {exit}
#
#pack .fmenuBar.quit -in .fmenuBar -side left
 
# File-menu---------------------------------------------------------
menubutton .fmenuBar.file -text "File" -menu .fmenuBar.file.m\
	-underline 0

menu .fmenuBar.file.m

.fmenuBar.file.m add command -label "Load (.ch2)" -command {\
	global cvupdate_while_sliding; \
	set cvupdate_while_sliding 0 ; \
	deletePolymer; readFile ; \
	update idletasks; set cvupdate_while_sliding 1;\
	createPolymer;\
	updateView NULL 0;\
	}\
	-underline 0 -accelerator Ctrl+l

.fmenuBar.file.m add command -label "Save (.ch2)" \
	-command "writeFile writeCH2 .ch2"\
	-underline 0 -accelerator Ctrl+s

.fmenuBar.file.m add separator

.fmenuBar.file.m add command -label "Quit" -command "exit"\
	-underline 0 -accelerator Ctrl+q

pack .fmenuBar.file -in .fmenuBar -side left

# Export-menu--------------------------------------------------------
menubutton .fmenuBar.export -text "Export" -menu .fmenuBar.export.m\
	-underline 0

menu .fmenuBar.export.m

.fmenuBar.export.m add command -label "VRML (.wrl)"\
	-command "write3DObject 1 .wrl"
.fmenuBar.export.m add command -label "TIFF (.tiff)"\
        -command "writeFile dumpScreen .tiff"
.fmenuBar.export.m add command -label "Wavefront (.obj)"\
        -command "write3DObject 2 .obj"

pack .fmenuBar.export -in .fmenuBar -side left

# Polymer-menu---------------------------------------------------------
menubutton .fmenuBar.polymer -text "Polymer" -menu .fmenuBar.polymer.m\
	-underline 0

menu .fmenuBar.polymer.m

.fmenuBar.polymer.m add radio -label "Single Chain" -variable TYPE\
	-command { if {$Type != 0 && $TA == 0} {
		      set Type 0;
		      updateView Type 0;
	              createCanvas $Lattice $Type; 
		      createRenderingbotton;
		      createViewbotton;
		   }\
		 }

.fmenuBar.polymer.m add radio -label "Liquid-like Region" -variable TYPE\
	-command { if {$Type != 1 && $TA == 0} {
		      set Type 1; 
		      updateView Type 1;
	              createCanvas $Lattice $Type; 
		      createRenderingbotton;
		      createViewbotton;
		   }\
		 }
.fmenuBar.polymer.m invoke 2

pack .fmenuBar.polymer -in .fmenuBar -side left

# Model-menu---------------------------------------------------------
menubutton .fmenuBar.model -text "Model" -menu .fmenuBar.model.m\
	-underline 0

menu .fmenuBar.model.m

.fmenuBar.model.m add radio -label "Cubic" -variable MODEL\
	-command {  if {$Lattice != 0 && $TA == 0} {
			set Lattice 0; 
		        updateView Lattice 0; 
		        createCanvas $Lattice $Type
			createViewbotton;
	 	    }\
		 }

.fmenuBar.model.m add radio -label "Diamond" -variable MODEL\
	-command {  if {$Lattice != 1 && $TA == 0} {
		       set Lattice 1; 
		       updateView Lattice 1; 
		       createCanvas $Lattice $Type 
		       createViewbotton;
		    }\
		  }
.fmenuBar.model.m invoke 2

pack .fmenuBar.model -in .fmenuBar -side left

# Rendering-menu---------------------------------------------------------
createRenderingbotton;

# View-menu---------------------------------------------------------
createViewbotton;

# Help-menu---------------------------------------------------------
menubutton .fmenuBar.help -text Help -menu .fmenuBar.help.m -underline 0

menu .fmenuBar.help.m

.fmenuBar.help.m add command -label "Tutorial" -command "helpTut" -underline 0

pack .fmenuBar.help -in .fmenuBar -side right

.fmenuBar.help.m add command -label "About Polymer Laboratory" \
-command "helpAbout" -underline 0

pack .fmenuBar.help -in .fmenuBar -side right

# Interact-menu---------------------------------------------------------
button .fmenuBar.interact -text Debug -command "wm deiconify .dbInteract"
pack .fmenuBar.interact -in .fmenuBar -side right 

# Teaching-Aid-menu---------------------------------------------------------
menubutton .fmenuBar.ta -text teaching-aids -menu .fmenuBar.ta.m -underline 0

menu .fmenuBar.ta.m
pack .fmenuBar.ta -in .fmenuBar -side right

.fmenuBar.ta.m add radio -label "polymer(s)" \
                -variable cvTA \
		-command {set TA 0; \
			  deletePolymer; \
			  catch {.3dwin.f3d.togl resetView}; \
			  teachingAid 0} 
.fmenuBar.ta.m invoke 1

.fmenuBar.ta.m add radio -label "cubic Lattice" \
                -variable cvTA \
                -command {set TA 1; \
			  deletePolymer; \
                          catch {.3dwin.f3d.togl resetView}; \
                          teachingAid 1}

.fmenuBar.ta.m add radio -label "diamond Lattice" \
                -variable cvTA \
                -command {set TA 2; \
			  deletePolymer; \
                          catch {.3dwin.f3d.togl resetView}; \
                          teachingAid 2}

.fmenuBar.ta.m add radio -label "trans" \
                -variable cvTA \
                -command {set TA 3; \
			  deletePolymer; \
                          catch {.3dwin.f3d.togl resetView}; \
                          teachingAid 3}

.fmenuBar.ta.m add radio -label "gauche+" \
                -variable cvTA \
                -command {set TA 4; \
			  deletePolymer; \
                          catch {.3dwin.f3d.togl resetView}; \
                          teachingAid 4}

.fmenuBar.ta.m add radio -label "gauche-" \
                -variable cvTA \
                -command {set TA 5; \
			  deletePolymer; \
                          catch {.3dwin.f3d.togl resetView}; \
                          teachingAid 5}

# Global KeyBindings for main-window--------------------------------
bind . <Control-KeyPress-q> {exit}
bind . <Control-KeyPress-l> {\
	global cvupdate_while_sliding; \
	deletePolymer; readFile ; \
	set cvupdate_while_sliding 0 ; \
	update idletasks; set cvupdate_while_sliding 1;\
	createPolymer }
bind . <Control-KeyPress-s> {writeFile}
bind . <KeyPress-F1> {helpTut}

createPolymer
open3D

# EOF

