static char SccsId[] = "@(#)render.c	1.1 06/24/97 polymer";

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <tcl.h>
#include <tk.h>
#include <togl.h>
#include "polymerl.h"


extern struct PLargs arg;
extern float *x, *y, *z;
extern int Color_Revert;
static GLfloat green_color[4] = { 0.0, 1.0, 0.0 };
static GLfloat blue_color[4] = { 0.0, 1.0, 1.0 };
static GLfloat red_color[4] = { 1.0, 0.0, 0.0 };
static GLfloat black_color[4] = { 0.0, 0.0, 0.0 };
static GLfloat white_color[4] = { 1.0, 1.0, 1.0 };
static GLfloat silver_color[4] = { 0.9, 0.9, 0.8 };

/********************************************/
/*drawpolymerchain:
  draw polymer chains as points and lines
  using OpenGL                              */
/********************************************/
void
drawpolymerchain ()
{
  int i, length, side;
  GLfloat ps = 4;

  srand(arg.seed);

  if (arg.lattice) {
      make_new_chain_diamond (0, 0, &side, &length);
      if (arg.ch2)
	 drawH_line(length);
  } 
  else	   
      make_new_chain_cubic (0, 0, &side, &length);

  /* points for C atoms */
  glPointSize(ps);
  glColor3fv (green_color);
  glDisable (GL_LIGHTING);
  glBegin (GL_POINTS);
    for (i = 1; i < length; i++) 
       glVertex3f (x[i], y[i], z[i]);
  glEnd ();
  glEnable (GL_LIGHTING);

  /* end points of the polymer chain */
  glPointSize(ps*2);
  glColor3fv(red_color);
  glDisable (GL_LIGHTING);
  glBegin (GL_POINTS);

    glVertex3f (x[0], y[0], z[0]);
    glVertex3f (x[length], y[length], z[length]);

  glEnd ();
  glEnable (GL_LIGHTING);

  /* lines for C bonds */
  glDisable (GL_LIGHTING);
  if ( Color_Revert ) 
     glColor3fv (black_color);
  else
     glColor3fv (white_color);
  glBegin (GL_LINES);
    for (i = 1; i <= length; i++) {
       glVertex3f (x[i-1], y[i-1], z[i-1]);
       glVertex3f (x[i], y[i], z[i]);
    }
  glEnd ();
  glEnable (GL_LIGHTING);
 
  return;
} /* drawpolymerchain */

/********************************************/
/*drawpolymerframes:
  draw polymer chains as dodecahedrons and 
    cylinders in wireframes
  using OpenGL                              */
/********************************************/
void
drawpolymerframes ()
{
  int i, length, side;
  GLdouble ct[3], delta[3];

  srand(arg.seed);

  if (arg.lattice) {
      make_new_chain_diamond (0, 0, &side, &length);
      if (arg.ch2)
         drawH_bonds(length);
  }
  else
      make_new_chain_cubic (0, 0, &side, &length);

  glDisable (GL_LIGHTING);
    /* first and last dodecahedron */
    glColor3fv (red_color);
    ct[0] = x[0]; 
    ct[1] = y[0]; 
    ct[2] = z[0]; 
    dodecahedron(ct, radius*2, GL_LINE_LOOP);
    ct[0] = x[length]; 
    ct[1] = y[length]; 
    ct[2] = z[length]; 
    dodecahedron(ct, radius*2, GL_LINE_LOOP);
    /* dodecahedrons */
    glColor3fv (silver_color);
    for (i = 1; i < length; i++) {
       ct[0] = x[i];
       ct[1] = y[i];
       ct[2] = z[i];
       dodecahedron(ct, radius*2, GL_LINE_LOOP);
    }
    /* cylinders */
    glColor3fv (silver_color);
    for (i = 0; i < length; i++) {
       ct[0] = x[i];
       ct[1] = y[i];
       ct[2] = z[i];
       delta[0] = x[i+1] - x[i];
       delta[1] = y[i+1] - y[i];
       delta[2] = z[i+1] - z[i];
       mycylinder (ct, radius, 4, delta, GL_LINE_STRIP);
    }
  glEnable (GL_LIGHTING);

  return;

}  /* drawpolymerframes */

/****************************************************/
/*drawpolymerbonds:
  draw polymer chains as dodecahedrons and 
      cylinders with flat shading or smooth shading
  using OpenGL                                      */
/****************************************************/
void
drawpolymerbonds ()
{
  int i, length, side;
  GLdouble ct[3], delta[3];
 
  srand(arg.seed);
 
  if (arg.lattice) {
      make_new_chain_diamond (0, 0, &side, &length);
      if (arg.ch2)
         drawH_bonds(length);
  }
  else
      make_new_chain_cubic (0, 0, &side, &length);
 
  if ( Color_Revert )
     glColor3fv (black_color);
  else
     glColor3fv (white_color);
  glEnable (GL_LIGHTING);
  for (i = 0; i <= length; i++) {
       ct[0] = x[i];
       ct[1] = y[i];
       ct[2] = z[i];
       delta[0] = x[i+1] - x[i];
       delta[1] = y[i+1] - y[i];
       delta[2] = z[i+1] - z[i];
       dodecahedron(ct, radius*2, GL_TRIANGLE_FAN);
       if (i < length)
	   mycylinder (ct, radius, 4, delta, GL_TRIANGLE_STRIP);
  }
  glDisable (GL_LIGHTING);
  return;
}

/********************************************/
/*drawpolymer chains in the amorphous region:
  using OpenGL                              */
/********************************************/
void
drawamorphous_region (int clip, int loop, int tie, 
		      int freeEnd, int doubleside)
{
   void render_planes(int, int, int);
   int i0, j0, k0, i, j, k, flag, length, side;
   int tmp;
   GLdouble ps = 2.5;

   render_planes(arg.Y, arg.Z, arg.M);

   if (clip)
      flag = 2;   /* amorphous region + clip */
   else
      flag = 1;   /* amorphous region ; single chain --- flag = 0 */

   tmp = arg.seed;
   for (k0 = 0; k0 <= doubleside; k0++) 
   for (i0 = -arg.Y+1; i0 < arg.Y; i0++) 
   for (j0 = -arg.Z+1; j0 < arg.Z; j0++) {

        /* try to get a relative random seed */
        srand(++tmp);
	for (i = 0; i < rand(); i++)
           rand();
        srand(rand());

	if (randfloat() < arg.D*0.01) {

           arg.x_init = 0; arg.y_init = i0; arg.z_init = j0; 

           if (arg.lattice) 
              make_new_chain_diamond (k0, flag, &side, &length);
	   else
	      make_new_chain_cubic (k0, flag, &side, &length);

           if ( (loop && side == 1) || 
	        (tie && side == 2) || 
	        (freeEnd && side == 4) ||
                (loop && tie && freeEnd) ) {

              /* end points of the polymer chain */
              glPointSize(ps*2);
              glColor3fv(red_color);
              glDisable (GL_LIGHTING);
              glBegin (GL_POINTS);
                glVertex3f (x[0], y[0], z[0]);

	        if (side >= 2) {
	           if (k0) 
	               glColor3fv(blue_color);
	           else	        		
	               glColor3fv(green_color);
	        }
	       
                glVertex3f (x[length], y[length], z[length]);
              glEnd ();
              glEnable (GL_LIGHTING);

              /* lines for C bonds */
              glDisable (GL_LIGHTING);
    	      if ( Color_Revert )
                 glColor3fv (black_color);
	      else
                 glColor3fv (white_color);
              glBegin (GL_LINES);

                for (i = 1; i <= length; i++) {
                   glVertex3f (x[i-1], y[i-1], z[i-1]);
                   glVertex3f (x[i], y[i], z[i]);
                }

              glEnd ();
              glEnable (GL_LIGHTING);
           }
        }
   }
}

/**********************************************************/
/* rendering the absorbing planes 			  */
/**********************************************************/
void render_planes(int NY, int NZ, int M)
{
   static GLdouble n0[3], d1[3], d2[3], d3[3], d4[3];

   d1[0] = 0.0; d1[1] = -NY; d1[2] = -NZ;
   d2[0] = 0.0; d2[1] = -NY; d2[2] = NZ;
   d3[0] = 0.0; d3[1] = NY; d3[2] = NZ;
   d4[0] = 0.0; d4[1] = NY; d4[2] = -NZ;
   if (arg.lattice) {
      d1[2] = d4[2] = -NZ*sqrt(2.0);
      d2[2] = d3[2] = NZ*sqrt(2.0);
   }
   n0[0] = -1.0; n0[1] = 0.0; n0[2] = 0.0;

   if ( Color_Revert )
      glColor3fv (black_color);
   else	
      glColor3fv (white_color);
   glDisable (GL_LIGHTING);
   glBegin (GL_LINE_LOOP);
     glVertex3dv(d1);
     glVertex3dv(d2);
     glVertex3dv(d3);
     glVertex3dv(d4);
   glEnd();

   if (arg.lattice) 
      d1[0] = d2[0] = d3[0] = d4[0] = M*sqrt(2.0);
   else
      d1[0] = d2[0] = d3[0] = d4[0] = M;

   n0[0] = 1.0; n0[1] = 0.0; n0[2] = 0.0;

   glBegin (GL_LINE_LOOP);
     glVertex3dv(d1);
     glVertex3dv(d2);
     glVertex3dv(d3);
     glVertex3dv(d4);
  glEnd();
  glEnable (GL_LIGHTING);
}

/********************************************/
/* my cylinder:
   Draws an cylinder with center at ct, num number of sides */
/********************************************/
void 
mycylinder (GLdouble ct[3], GLdouble sc, GLint num, 
	    GLdouble a[3], GLenum type)
{
   GLdouble d1[3], d2[3], d3[3], d4[3];
   int j;
   struct cylinder pts;
 
   Points(sc, &pts, num, ct, a);
  
   for (j = 0; j < num; j++) {
      d1[0] = pts.topx[j]; 
      d1[1] = pts.topy[j]; 
      d1[2] = pts.topz[j];
 
      d2[0] = pts.downx[j]; 
      d2[1] = pts.downy[j]; 
      d2[2] = pts.downz[j];
 
      d3[0] = pts.topx[j+1]; 
      d3[1] = pts.topy[j+1]; 
      d3[2] = pts.topz[j+1];

      d4[0] = pts.downx[j+1]; 
      d4[1] = pts.downy[j+1]; 
      d4[2] = pts.downz[j+1];
 
      if (type == GL_LINE_STRIP) {
         glBegin (type);
           glVertex3dv(d1); glVertex3dv(d2);
           glVertex3dv(d4); glVertex3dv(d3);
         glEnd();
      }	
      else
          myquad (d1, d2, d3, d4, ct, type); 
   }
}
 
/********************************************/
/*  myquad:
      rendering quadrilateral		    */ 
/********************************************/
void 
myquad (GLdouble d1[3], GLdouble d2[3], GLdouble d3[3], 
        GLdouble d4[3], GLdouble ct[3], GLenum shadeType)
{
   GLdouble n1[3], n2[3], n3[3], n4[3];
 
   glBegin (shadeType);
     if (arg.mode == 3) { /* smooth shading */
        diff3 (ct, d1, n1); diff3 (ct, d2, n2);
        diff3 (ct, d3, n3); diff3 (ct, d4, n4);
        normalize(n1); normalize(n2);
        normalize(n3); normalize(n4);

        glNormal3dv(n1); glVertex3dv(d1);
        glNormal3dv(n2); glVertex3dv(d2);
        glNormal3dv(n3); glVertex3dv(d3);
        glNormal3dv(n4); glVertex3dv(d4);
     }
     else {	/* flat shading */
        diff3 (d1, d2, n1); 
        diff3 (d2, d3, n2); 
        crossprod(n1, n2, n3);
        normalize(n3);
	glNormal3dv(n3);

	glVertex3dv(d1); glVertex3dv(d2);
        glVertex3dv(d3); glVertex3dv(d4);
     }

   glEnd();
}

/********************************************/
/* initdodec:
     vertex of decahedron		    */
/********************************************/
void 
initdodec(GLdouble dodec[20][3])
{
    GLdouble alpha, beta;

    alpha = sqrt(2.0/(3.0 + sqrt(5.0)));
    beta = 1.0 + sqrt(6.0/(3.0 + sqrt(5.0)) - 2.0 + 2.0*sqrt(2.0/(3.0 +
							    sqrt(5.0))));
    dodec[0][0] = -alpha; dodec[0][1] = 0; dodec[0][2] = beta;
    dodec[1][0] = alpha; dodec[1][1] = 0; dodec[1][2] = beta;
    dodec[2][0] = -1; dodec[2][1] = -1; dodec[2][2] = -1;
    dodec[3][0] = -1; dodec[3][1] = -1; dodec[3][2] = 1;
    dodec[4][0] = -1; dodec[4][1] = 1; dodec[4][2] = -1;
    dodec[5][0] = -1; dodec[5][1] = 1; dodec[5][2] = 1;
    dodec[6][0] = 1; dodec[6][1] = -1; dodec[6][2] = -1;
    dodec[7][0] = 1; dodec[7][1] = -1; dodec[7][2] = 1;
    dodec[8][0] = 1; dodec[8][1] = 1; dodec[8][2] = -1;
    dodec[9][0] = 1; dodec[9][1] = 1; dodec[9][2] = 1;
    dodec[10][0] = beta; dodec[10][1] = alpha; dodec[10][2] = 0;
    dodec[11][0] = beta; dodec[11][1] = -alpha; dodec[11][2] = 0;
    dodec[12][0] = -beta; dodec[12][1] = alpha; dodec[12][2] = 0;
    dodec[13][0] = -beta; dodec[13][1] = -alpha; dodec[13][2] = 0;
    dodec[14][0] = -alpha; dodec[14][1] = 0; dodec[14][2] = -beta;
    dodec[15][0] = alpha; dodec[15][1] = 0; dodec[15][2] = -beta;
    dodec[16][0] = 0; dodec[16][1] = beta; dodec[16][2] = alpha;
    dodec[17][0] = 0; dodec[17][1] = beta; dodec[17][2] = -alpha;
    dodec[18][0] = 0; dodec[18][1] = -beta; dodec[18][2] = alpha;
    dodec[19][0] = 0; dodec[19][1] = -beta; dodec[19][2] = -alpha;
}

/********************************************/
/* dodecahedron:
   Draws an dodecahedron with center at 0.0. The radius
   is sqrt(3).  */
/********************************************/
void 
dodecahedron(GLdouble ct[3], GLdouble sc, GLenum type)
{
    GLdouble dodec[20][3];
    void pentagon(GLdouble [3], GLdouble, 
		  GLdouble [3], GLdouble [3], GLdouble [3], 
		  GLdouble [3], GLdouble [3], GLenum);
    void initdodec(GLdouble dodec[20][3]);

    initdodec(dodec);
    pentagon(ct, sc,  &dodec[0][0], &dodec[1][0], &dodec[9][0], 
		   &dodec[16][0], &dodec[5][0], type);
    pentagon(ct, sc,  &dodec[1][0], &dodec[0][0], &dodec[3][0], 
		   &dodec[18][0], &dodec[7][0], type);
    pentagon(ct, sc,  &dodec[1][0], &dodec[7][0], &dodec[11][0], 
		   &dodec[10][0], &dodec[9][0], type);
    pentagon(ct, sc,  &dodec[11][0], &dodec[7][0], &dodec[18][0], 
		   &dodec[19][0], &dodec[6][0], type);
    pentagon(ct, sc,  &dodec[8][0], &dodec[17][0], &dodec[16][0], 
		   &dodec[9][0], &dodec[10][0], type);
    pentagon(ct, sc,  &dodec[2][0], &dodec[14][0], &dodec[15][0], 
		   &dodec[6][0], &dodec[19][0], type);
    pentagon(ct, sc,  &dodec[2][0], &dodec[13][0], &dodec[12][0], 
		   &dodec[4][0], &dodec[14][0], type);
    pentagon(ct, sc,  &dodec[2][0], &dodec[19][0], &dodec[18][0], 
		   &dodec[3][0], &dodec[13][0], type);
    pentagon(ct, sc,  &dodec[3][0], &dodec[0][0], &dodec[5][0], 
		   &dodec[12][0], &dodec[13][0], type);
    pentagon(ct, sc,  &dodec[6][0], &dodec[15][0], &dodec[8][0], 
	  	   &dodec[10][0], &dodec[11][0], type);
    pentagon(ct, sc,  &dodec[4][0], &dodec[17][0], &dodec[8][0], 
		   &dodec[15][0], &dodec[14][0], type);
    pentagon(ct, sc,  &dodec[4][0], &dodec[12][0], &dodec[5][0], 
		   &dodec[16][0], &dodec[17][0], type);
}

/********************************************/
/* pentagon:
      rendering pentagon		    */
/********************************************/
void 
pentagon(GLdouble ct[3], GLdouble sc, GLdouble v1[3], GLdouble v2[3], 
	GLdouble v3[3], GLdouble v4[3], GLdouble v5[3], GLenum shadeType)
{
    GLdouble n0[3], n1[3],  nout[3], d1[3], d2[3], d3[3], d4[3], d5[3];

    scalarmult(sc, v1, d1);
    scalarmult(sc, v2, d2);
    scalarmult(sc, v3, d3);
    scalarmult(sc, v4, d4);
    scalarmult(sc, v5, d5);

    glBegin (shadeType);

    if (shadeType == GL_LINE_LOOP) {
        add3 (d1, ct, d1); glVertex3dv(d1);
        add3 (d2, ct, d2); glVertex3dv(d2);
        add3 (d3, ct, d3); glVertex3dv(d3);
        add3 (d4, ct, d4); glVertex3dv(d4);
        add3 (d5, ct, d5); glVertex3dv(d5);
    }   
    else if (arg.mode == 2) {
	diff3(d1,d2,n0); diff3(d2,d3,n1);
	crossprod(n0, n1, nout);
        normalize(nout);

        glNormal3dv(nout);

        add3 (d1, ct, d1); glVertex3dv(d1);
        add3 (d2, ct, d2); glVertex3dv(d2);
        add3 (d3, ct, d3); glVertex3dv(d3);
        add3 (d4, ct, d4); glVertex3dv(d4);
        add3 (d5, ct, d5); glVertex3dv(d5);
    }
    else {
	copy3(d1, nout); normalize(nout); glNormal3dv(nout);
        add3 (d1, ct, d1); glVertex3dv(d1);
        copy3(d2, nout); normalize(nout); glNormal3dv(nout);
        add3 (d2, ct, d2); glVertex3dv(d2);
        copy3(d3, nout); normalize(nout); glNormal3dv(nout);
        add3 (d3, ct, d3); glVertex3dv(d3);
        copy3(d4, nout); normalize(nout); glNormal3dv(nout);
        add3 (d4, ct, d4); glVertex3dv(d4);
        copy3(d5, nout); normalize(nout); glNormal3dv(nout);
        add3 (d5, ct, d5); glVertex3dv(d5);
    }
    glEnd();
}
/********************************************/
/* points:
      calculate the coords of points on my cylinder */
/********************************************/
void 
Points (float rd, struct cylinder* pts,
        int pn, GLdouble cpt[3], GLdouble a[3])
{
   int i;
   GLdouble b[3], w[3];
   float alpha;
 
   if (pn > 17) {
      fprintf(stderr, "too many points!\n");
      exit (1);
   }
   if (abs(a[0]*a[1]*a[2]) > 0.0001) {
      b[0] = 0; b[1] = a[2]; b[2] = -a[1];
   }
   else if (abs(a[0]) < 0.001) {
      b[0] = 0; b[1] = a[2]; b[2] = -a[1];
   }
   else if (abs(a[1]) < 0.001) {
      b[0] = -a[2]; b[1] = 0; b[2] = a[0];
   }
   else {
      b[0] = a[1]; b[1] = -a[0]; b[2] = 0;
   }
   normalize(b);
 
   crossprod(a, b, w);
   normalize(w);
 
   for (i = 0; i < pn; i++) {
      alpha = 2*3.1415926*i/pn;
      pts->topx[i] = sin(alpha)*w[0]*rd + cos(alpha)*b[0]*rd + cpt[0];
      pts->topy[i] = sin(alpha)*w[1]*rd + cos(alpha)*b[1]*rd + cpt[1];
      pts->topz[i] = sin(alpha)*w[2]*rd + cos(alpha)*b[2]*rd + cpt[2];
      pts->downx[i] = pts->topx[i] + a[0];
      pts->downy[i] = pts->topy[i] + a[1];
      pts->downz[i] = pts->topz[i] + a[2];
   }
   pts->topx[pn] = pts->topx[0];
   pts->topy[pn] = pts->topy[0];
   pts->topz[pn] = pts->topz[0];
 
   pts->downx[pn] = pts->downx[0];
   pts->downy[pn] = pts->downy[0];
   pts->downz[pn] = pts->downz[0];
 
}
