/**
 * A best exemplar finder.  Scans over the entire image (using a
 * sliding window) and finds the exemplar which minimizes the sum
 * squared error (SSE) weighted by confidence term over the 
 * to-be-filled pixels in the target patch. 
 *
 * @author Sooraj Bhat
 * Modified by Qiangwei Peng on
 * 23/3/27 to change the similarity measure from SSD to CSSD. 
 */
#include "mex.h"
#include <limits.h>

void bestexemplarhelper_CSSD(const int mm, const int nn, const int m, const int n, 
			const double *img, const double *Ip, 
			const mxLogical *toFill, const mxLogical *sourceRegion, const double *patch_C, 
			double *best) 
{
  register int i,j,ii,jj,ii2,jj2,M,N,I,J,ndx,ndx2,mn=m*n,mmnn=mm*nn;
  double patchErr=0.0,err=0.0,err_3d=0.0,weight=0.0,bestErr=1000000000.0;

  /* foreach patch */
  N=nn-n+1;  M=mm-m+1;
  for (j=1; j<=N; ++j) {
    J=j+n-1;
    for (i=1; i<=M; ++i) {
      I=i+m-1;
      /*** Calculate patch error ***/
      /* foreach pixel in the current patch */
      for (jj=j,jj2=1; jj<=J; ++jj,++jj2) {
	for (ii=i,ii2=1; ii<=I; ++ii,++ii2) {
	  ndx=ii-1+mm*(jj-1);
	  if (!sourceRegion[ndx])
	    goto skipPatch;
	  ndx2=ii2-1+m*(jj2-1);
	  err_3d=0.0;
	  weight = patch_C[ndx2];
	  err=img[ndx      ] - Ip[ndx2    ]; err_3d += err*err;
	  err=img[ndx+=mmnn] - Ip[ndx2+=mn]; err_3d += err*err;
	  err=img[ndx+=mmnn] - Ip[ndx2+=mn]; err_3d += err*err;
	  patchErr += err_3d * weight * weight;
	}
      }
      /*** Update ***/
      if (patchErr < bestErr) {
	bestErr = patchErr; 
	best[0] = i; best[1] = I;
	best[2] = j; best[3] = J;
      }
      /*** Reset ***/
    skipPatch:
      patchErr = 0.0; 
    }
  }
}

/* best = bestexemplarhelper_CSSD(mm,nn,m,n,img,Ip,toFill,sourceRegion,patch_C); */
void mexFunction(int nlhs,mxArray *plhs[],int nrhs,const mxArray *prhs[]) 
{
  int mm,nn,m,n;
  double *img,*Ip,*patch_C,*best;
  mxLogical *toFill,*sourceRegion;

  /* Extract the inputs */
  mm = (int)mxGetScalar(prhs[0]);
  nn = (int)mxGetScalar(prhs[1]);
  m  = (int)mxGetScalar(prhs[2]);
  n  = (int)mxGetScalar(prhs[3]);
  img = mxGetPr(prhs[4]);
  Ip  = mxGetPr(prhs[5]);
  toFill = mxGetLogicals(prhs[6]);
  sourceRegion = mxGetLogicals(prhs[7]);
  patch_C = mxGetPr(prhs[8]);
  
  /* Setup the output */
  plhs[0] = mxCreateDoubleMatrix(4,1,mxREAL);
  best = mxGetPr(plhs[0]);
  best[0]=best[1]=best[2]=best[3]=0.0;

  /* Do the actual work */
  bestexemplarhelper_CSSD(mm,nn,m,n,img,Ip,toFill,sourceRegion,patch_C,best);
}
