function [D, D_thr, dR] = determine_d_thr(AfterIARPGimgFilename,fillFilename)
% INPAINT_2ND_RGB  The second stage of the integration image inpainting 
% algorithm for RGB image of the paper:
%
% Qiangwei Peng and Wen Huang,
% An Image Inpainting Algorithm Using Exemplar Matching and Low-Rank Sparse
% Prior
%
% Usage:   [inpaintedImg,origImg,C,D,fillMovie] = inpaint_2nd...
%             (AfterIARPGimgFilename,fillFilename,HalfPatchSize,UseGraphut)
% Inputs: 
%   AfterIARPGimgFilename   Filename of the IARPG method output results.
%   fillFilename            Filename of the binary mask specifying the fill
%                           region. 
%   HalfPatchSize           The radius of patch size, i.e. patch size = 2 × 
%                           HalfPatchSize + 1
%   UseGraphut              Whether to use graph cut algorithm, 1 means yes
%                           , 0 means no.
%
% Outputs:
%   inpaintedImg   The inpainted image; an MxN×3 matrix of doubles. 
%   origImg        The original image; an MxN×3 matrix of doubles.
%   C              MxN matrix of confidence values accumulated over all iterations.
%   D              MxN matrix of data term values accumulated over all iterations.
%   fillMovie      A Matlab movie struct depicting the fill region over time. 
%
% Example:
%   [i1,i2,c,d,mov] = inpaint('iarpg1.png','mask1.png',4,1);
%   imshow(uint8(i1)) % show inpainted result
%   movie(gcf,mov); % grab some popcorn
%
%   author: Qiangwei Peng on 22/25/6
%   This code is based on the initial implementation of Criminisi algorithm
%   by Sooraj Bhat and the modifications made by Marcel Davey and John Gu.
   

[img,fillRegion] = loadimgs(AfterIARPGimgFilename,fillFilename);
img = double(img);
sz = [size(img,1) size(img,2)];

% Initialize isophote values
[Ix(:,:,3), Iy(:,:,3)] = gradient(img(:,:,3));
[Ix(:,:,2), Iy(:,:,2)] = gradient(img(:,:,2));
[Ix(:,:,1), Iy(:,:,1)] = gradient(img(:,:,1));
Ix = sum(Ix,3)/(3*255); Iy = sum(Iy,3)/(3*255);
temp = Ix; Ix = -Iy; Iy = temp;  % Rotate gradient 90 degrees

% Initialize data terms
D = repmat(-.1,sz);
D = nan(sz);


% Find contour & normalized gradients of fill region
fillRegionD = double(fillRegion); % Marcel 11/30/05
dR = find(conv2(fillRegionD,[1,1,1;1,-8,1;1,1,1],'same')>0); % Marcel 11/30/05

[Nx,Ny] = gradient(double(~fillRegion)); % Marcel 11/30/05
N = [Nx(dR(:)) Ny(dR(:))];
N = normr(N);
N(~isfinite(N))=0; % handle NaN and Inf

% Compute patch priorities = confidence term * data term
D(dR) = abs(Ix(dR).*N(:,1)+Iy(dR).*N(:,2)) + 0.001;

% determine D_thr
temp = D(dR);
[idx,~] = kmeans(temp,2);
D_thr = sqrt(mean(temp(idx==1))*mean(temp(idx==2)));

%---------------------------------------------------------------------
% Loads the an image and it's fill region.
%---------------------------------------------------------------------
function [img,fillRegion] = loadimgs(imgFilename,fillFilename)
img = imread(imgFilename); 
fillRegion = logical(imread(fillFilename));
end

function [A] = normr(N)
    for ii=1:size(N,1)
        A(ii,:) = N(ii,:)/norm(N(ii,:));
    end
end

end