function [inpaintedImg,origImg,C,D,fillMovie] = inpaint_2nd_rgb(AfterIARPGimgFilename,fillFilename,HalfPatchSize,UseGraphcut)
% INPAINT_2ND_RGB  The second stage of the integration image inpainting 
% algorithm for RGB image of the paper:
%
% Qiangwei Peng and Wen Huang,
% An Image Inpainting Algorithm Using Exemplar Matching and Low-Rank Sparse
% Prior
%
% Usage:   [inpaintedImg,origImg,C,D,fillMovie] = inpaint_2nd...
%             (AfterIARPGimgFilename,fillFilename,HalfPatchSize,UseGraphut)
% Inputs: 
%   AfterIARPGimgFilename   Filename of the IARPG method output results.
%   fillFilename            Filename of the binary mask specifying the fill
%                           region. 
%   HalfPatchSize           The radius of patch size, i.e. patch size = 2 × 
%                           HalfPatchSize + 1
%   UseGraphut              Whether to use graph cut algorithm, 1 means yes
%                           , 0 means no.
%
% Outputs:
%   inpaintedImg   The inpainted image; an MxN×3 matrix of doubles. 
%   origImg        The original image; an MxN×3 matrix of doubles.
%   C              MxN matrix of confidence values accumulated over all iterations.
%   D              MxN matrix of data term values accumulated over all iterations.
%   fillMovie      A Matlab movie struct depicting the fill region over time. 
%
% Example:
%   [i1,i2,c,d,mov] = inpaint('iarpg1.png','mask1.png',4,1);
%   imshow(uint8(i1)) % show inpainted result
%   movie(gcf,mov); % grab some popcorn
%
%   author: Qiangwei Peng on 22/25/6
%   This code is based on the initial implementation of Criminisi algorithm
%   by Sooraj Bhat and the modifications made by Marcel Davey and John Gu.
   

[img,fillRegion] = loadimgs(AfterIARPGimgFilename,fillFilename);
img = double(img);
origImg = img; 
ind = img2ind(img);
sz = [size(img,1) size(img,2)];
sourceRegion = ~fillRegion;

% Initialize isophote values
[Ix(:,:,3), Iy(:,:,3)] = gradient(img(:,:,3));
[Ix(:,:,2), Iy(:,:,2)] = gradient(img(:,:,2));
[Ix(:,:,1), Iy(:,:,1)] = gradient(img(:,:,1));
Ix = sum(Ix,3)/(3*255); Iy = sum(Iy,3)/(3*255);
temp = Ix; Ix = -Iy; Iy = temp;  % Rotate gradient 90 degrees

% Initialize confidence and data terms
C = double(sourceRegion);
D = repmat(-.1,sz);

% Visualization stuff
if nargout==5
  fillMovie(1).cdata=uint8(img); 
  fillMovie(1).colormap=[];
  origImg(1,1,:) = [0 0 0];
end

iter = 1;
% Loop until entire fill region has been covered
while any(fillRegion(:))
  % Find contour & normalized gradients of fill region
  fillRegionD = double(fillRegion); % Marcel 11/30/05
  dR = find(conv2(fillRegionD,[1,1,1;1,-8,1;1,1,1],'same')>0); % Marcel 11/30/05
  
  [Nx,Ny] = gradient(double(~fillRegion)); % Marcel 11/30/05
  N = [Nx(dR(:)) Ny(dR(:))];
  N = normr(N);  
  N(~isfinite(N))=0; % handle NaN and Inf
  
  % Compute confidences along the fill front
  for k=dR'
    Hp = getpatch(sz,k,HalfPatchSize);
    q = Hp(~(fillRegion(Hp)));
    C(k) = sum(C(q))/numel(Hp);
  end
  
  % Compute patch priorities = confidence term * data term
  D(dR) = abs(Ix(dR).*N(:,1)+Iy(dR).*N(:,2)) + 0.001;
  priorities = C(dR).* D(dR);  
  
  % determine D_thre
  if iter == 1
      temp = D(dR);
      [idx,~] = kmeans(temp,2);
      D_thr = sqrt(mean(temp(idx==1))*mean(temp(idx==2)));
  end
  
  % Find patch with maximum priority, Hp
  [~,ndx] = max(priorities(:));
  p = dR(ndx(1));
  [Hp,rows,cols] = getpatch(sz,p,HalfPatchSize);
  toFill = fillRegion(Hp);
  
  % Update confidence term
  C(Hp(toFill)) = C(p);
  patch_C = C(Hp);
  
  % Find exemplar that minimizes error, Hq
  Hq = bestexemplar(img,img(rows,cols,:),sourceRegion,D(p),D_thr,toFill',patch_C');

  % Update fill region
  toFill = logical(toFill);                 % Marcel 11/30/05
  fillRegion(Hp(toFill)) = false;
  
  % Whether to perform graph cut algorithm when copying pixels from Hq to Hp
  if UseGraphcut == 1
    isRGB = 1;
    toFill = graphcut(Hp,Hq,toFill,floor(HalfPatchSize/2),img,isRGB);  
  end
  
  % Propagate confidence & isophote values
  Ix(Hp(toFill)) = Ix(Hq(toFill));
  Iy(Hp(toFill)) = Iy(Hq(toFill));
  
  % Copy image data from Hq to Hp
  ind(Hp(toFill)) = ind(Hq(toFill));
  img(rows,cols,:) = ind2img(ind(rows,cols),origImg);

  % Visualization stuff
  if nargout==5
    ind2 = ind;
    ind2(logical(fillRegion)) = 1;          % Marcel 11/30/05
    fillMovie(iter+1).cdata=uint8(ind2img(ind2,origImg)); 
    fillMovie(iter+1).colormap=[];
  end
  iter = iter+1;
  
%   % If you want to see the inpainting process in real time, you can uncomment the code. 
%   imshow(uint8(img.*(~fillRegion)))
%   hold on
%   drawbox;

end

inpaintedImg=img;


%---------------------------------------------------------------------
% Scans over the entire image (with a sliding window)
% for the exemplar with the lowest error.
%---------------------------------------------------------------------
function Hq = bestexemplar(img,Ip,sourceRegion,Dp,D_thr,toFill,patch_C)
m=size(Ip,1); mm=size(img,1); n=size(Ip,2); nn=size(img,2);

% according to whether the current patch is texture patch or structure patch,
% different search methods for candidate patches are selected. 
if Dp > D_thr
    best = bestexemplarhelper_CSSD(mm,nn,m,n,img,Ip,toFill,sourceRegion,patch_C);
else
    best = bestexemplarhelper(mm,nn,m,n,img,Ip,toFill,sourceRegion);
end
Hq = sub2ndx(best(1):best(2),(best(3):best(4))',mm);


%---------------------------------------------------------------------
% Returns the indices for a patch centered at pixel p.
%---------------------------------------------------------------------
function [Hp,rows,cols] = getpatch(sz,p,HalfPatchSize)
% [x,y] = ind2sub(sz,p);  % 2*HalfPatchSize+1 == the patch size
w=HalfPatchSize; p=p-1; y=floor(p/sz(1))+1; p=rem(p,sz(1)); x=floor(p)+1;    
rows = max(x-w,1):min(x+w,sz(1));
cols = (max(y-w,1):min(y+w,sz(2)))';
Hp = sub2ndx(rows,cols,sz(1));


%---------------------------------------------------------------------
% Converts the (rows,cols) subscript-style indices to Matlab index-style
% indices.  Unforunately, 'sub2ind' cannot be used for this.
%---------------------------------------------------------------------
function N = sub2ndx(rows,cols,nTotalRows)
X = rows(ones(length(cols),1),:);
Y = cols(:,ones(1,length(rows)));
N = X+(Y-1)*nTotalRows;


%---------------------------------------------------------------------
% Converts an indexed image into an RGB image, using 'img' as a colormap
%---------------------------------------------------------------------
function img2 = ind2img(ind,img)
for i=3:-1:1, temp=img(:,:,i); img2(:,:,i)=temp(ind); end


%---------------------------------------------------------------------
% Converts an RGB image into a indexed image, using the image itself as
% the colormap.
%---------------------------------------------------------------------
function ind = img2ind(img)
s=size(img); ind=reshape(1:s(1)*s(2),s(1),s(2));


%---------------------------------------------------------------------
% Loads the an image and it's fill region.
%---------------------------------------------------------------------
function [img,fillRegion] = loadimgs(imgFilename,fillFilename)
img = imread(imgFilename); 
fillRegion = logical(imread(fillFilename));


function [A] = normr(N)
    for ii=1:size(N,1)
        A(ii,:) = N(ii,:)/norm(N(ii,:));
    end






