%% ================= User Selection =================
% mode_run = 1: First run (compare all algorithms -- Section 5.4)
% mode_run = 2: Second run (only compare RPG, RAPG, AR-RAPG -- Section 5.3)

mode_run = 1;   % <<== User manually change this (1 or 2)

%% ================= Problem setup =================
seed = 456;
rng(seed);

n = 200; m = 20; p = 4; lambda = 1;

A = randn(m, n);
A = A - repmat(mean(A, 1), m, 1);
A = A ./ repmat(sqrt(sum(A .* A)), m, 1);

[U, S, V] = svd(A, 'econ');
PCAV = V(:, 1:p);
D = diag(S(1:p, 1:p));
Dsq = D.^2;
Xinitial.main = PCAV;

Lu = norm(Dsq)^2 * 8;
tol = 1e-10*n*p;
maxiter = 10000;
A0 = 0.001; xi = 1; theta = 1; mu = 1; rho = 0.5;

%% ================= Run based on mode =================
if mode_run == 1
    %% -------- First run: Ll = Lu/4, all algorithms --------
    Ll = Lu / 4;
    q = (mu - rho) / (theta * Ll - rho);

    % Run all algorithms
    % RPG
    [xopt1, iter, time1, fv1, nD, sparsity1, avar, fs1, neta1] = RPG(Xinitial, A, Dsq, lambda, Ll, tol, maxiter);
    % ManPG
    [xopt2, iter, time2, fv2, nD, sparsity2, avar, fs2, neta2] = ManPG(Xinitial, A, Dsq, lambda, Ll, tol, maxiter);
    % ManPG-Ada
    [xopt3, iter, time3, fv3, nD, sparsity3, avar, fs3, neta3] = ManPG_Ada(Xinitial, A, Dsq, lambda, Ll, tol, maxiter, fv1);
    % RAPG
    [xopt4, iter, time4, fv4, nD, sparsity4, avar, fs4, neta4] = RAPG(Xinitial, A, Dsq, lambda, A0, xi, q, Ll, Lu, tol, maxiter, fv1);
    % AR-RAPG
    [xopt5, iter, time5, fv5, nD, sparsity5, avar, fs5, neta5] = AR_RAPG(Xinitial, A, Dsq, lambda, A0, xi, q, Ll, Lu, tol, maxiter, fv1);

    % Plot all 5 algorithms
    plot_convergence({fs1, fs2, fs3, fs4, fs5}, {neta1, neta2, neta3, neta4, neta5}, ...
                     {'RPG','ManPG','ManPG-Ada','RAPG','AR-RAPG'}, Ll, mode_run);

elseif mode_run == 2
    %% -------- Second run: Ll = Lu/4 & Lu/8*1.2, only RPG, RAPG, AR-RAPG --------
    Ll_list = [Lu/4, Lu/8*1.2];
    for k = 1:length(Ll_list)
        Ll = Ll_list(k);
        q = (mu - rho) / (theta * Ll - rho);

        % Only RPG, RAPG, AR-RAPG
        [xopt1, iter, time1, fv1, nD, sparsity1, avar, fs1, neta1] = RPG(Xinitial, A, Dsq, lambda, Ll, tol, maxiter);        
    
        [xopt4, iter, time4, fv4, nD, sparsity4, avar, fs4, neta4] = RAPG(Xinitial, A, Dsq, lambda, A0, xi, q, Ll, Lu, tol, maxiter, fv1);
    
        [xopt5, iter, time5, fv5, nD, sparsity5, avar, fs5, neta5] = AR_RAPG(Xinitial, A, Dsq, lambda, A0, xi, q, Ll, Lu, tol, maxiter, fv1);

        % Plot only selected algorithms
        plot_convergence({fs1, fs4, fs5}, {neta1, neta4, neta5}, {'RPG','RAPG','AR-RAPG'}, Ll, mode_run);
    end
else
    error('Invalid mode_run. Choose 1 or 2.');
end

%% ================= Plot function =================

function plot_convergence(fs_all, neta_all, labels, Ll, mode_run)
    start_eta = 2; gap_eta = 1; % gap_eta = 40;
    start_f   = 5; gap_f   = 1; % gap_f   = 20;
    lw = 1.8; 
    ms = 8;
    
    markevery = 200;

%     ms ={8,8,8,8,8};

%     colors = lines(length(fs_all));
%     markers = {'o','s','d','^','v','>','<','p','h'};
    %% k - ||eta_k||
    figure('Position',[300 300 500 300]); hold on;
    for i = 1:length(neta_all)
        [c, m, s] = get_style(labels{i}); % get color, marker, line style
        y = neta_all{i};
        idx = start_eta:gap_eta:length(y);
        numMarkers = 5;  % Total number of markers to show
        indices = round(linspace(1, length(idx), numMarkers));  % Calculate indices for evenly distributed markers
        if mode_run == 1
        semilogy(idx, y(idx), ...
            'Color', c, 'Marker', m, 'LineStyle', s, ...
            'LineWidth', lw, 'MarkerSize', ms, ...
            'MarkerIndices', indices, 'DisplayName', labels{i});
        else
        semilogy(idx, y(idx), ...
            'Color', c, 'Marker', m, 'LineStyle', s, ...
            'LineWidth', lw, 'MarkerSize', ms, ...
            'MarkerIndices', 1:markevery:length(idx), 'DisplayName', labels{i});
        end
    end
    xlabel('Iteration', 'FontSize', 20); ylabel('$\|\eta\|$','Interpreter','latex', 'FontSize', 20);
    legend('show'); grid off; box on;
    set(gca,'YScale','log','TickLabelInterpreter','latex');
    % Set font size for axis tick labels (numbers on axes)
    ax = gca;  % Get current axes handle
    ax.XAxis.FontSize = 12;  % x-axis tick labels font size
    ax.YAxis.FontSize = 12;  % y-axis tick labels font size

    if mode_run == 1
        xlim([0, 5000]); 
    elseif mode_run == 2
        xlim([0, 1000]);
    end
    filename_eta = fullfile('figure', ['eta_mode_' num2str(mode_run) '_Ll_' num2str(Ll) ]);
    saveas(gcf, [filename_eta '.eps'], 'epsc');
    saveas(gcf, [filename_eta '.pdf']);

%     title(['Convergence of $\|\eta_k\|$, Ll = ', num2str(Ll)], 'Interpreter','latex');

    %% k - F(x_k)
    figure('Position',[900 300 500 300]); hold on;
    for i = 1:length(fs_all)
        [c, m, s] = get_style(labels{i});
        y = fs_all{i};
        if mode_run == 1 && i == 1 
            idx = start_f:gap_f:length(y)/4;
        elseif mode_run == 1 && i == 2
            idx = start_f:gap_f:length(y)/4.2;
%         if mode_run == 1 && i == 1 || i == 2
%             idx = start_f:gap_f:length(y)/5;
        else
            idx = start_f:gap_f:length(y);
        end
        numMarkers = 5;  % Total number of markers to show
        indices = round(linspace(1, length(idx), numMarkers));  % Calculate indices for evenly distributed markers
        if mode_run == 1
            semilogy(idx, y(idx), ...
                'Color', c, 'Marker', m, 'LineStyle', s, ...
                'LineWidth', lw, 'MarkerSize', ms, ...
                'MarkerIndices', indices, 'DisplayName', labels{i});
        else
            semilogy(idx, y(idx), ...
                'Color', c, 'Marker', m, 'LineStyle', s, ...
                'LineWidth', lw, 'MarkerSize', ms, ...
                'MarkerIndices', 1:markevery:length(idx), 'DisplayName', labels{i});
        end
    end
    xlabel('Iteration', 'FontSize', 20); ylabel('F(X)', 'FontSize', 20);
    legend('show'); grid off; box on;
    ax = gca;  % Get current axes handle
    ax.XAxis.FontSize = 12;  % x-axis tick labels font size
    ax.YAxis.FontSize = 12;  % y-axis tick labels font size

    if mode_run == 1
        xlim([0, 2000]); ylim([39,46.5]);
    elseif mode_run == 2
        xlim([0, 1000]); ylim([39,47]);
    end
    filename_F = fullfile('figure', ['F_mode_' num2str(mode_run) '_Ll_' num2str(Ll) ]);
    saveas(gcf, [filename_F '.eps'], 'epsc');
    saveas(gcf, [filename_F '.pdf']);

%     title(['Convergence of F(X), Ll = ', num2str(Ll)]);
end

%% Fixed color, marker, and line style for each algorithm
function [color, marker, style] = get_style(name)
    switch name
        case 'RPG'
            color = [0 0.4470 0.7410];   % blue
            marker = 'o'; style = '-';
        case 'ManPG'
            color = [0.8500 0.3250 0.0980]; % orange
            marker = '<'; style = '-';
        case 'ManPG-Ada'
            color = [0.9290 0.6940 0.1250]; % yellow
            marker = 's'; style = '-';
        case 'RAPG'
            color = [0.4940 0.1840 0.5560]; % purple
            marker = '^'; style = '-';
        case 'AR-RAPG'
            color = [0.4660 0.6740 0.1880]; % green
            marker = 'v'; style = '-';
        otherwise
            color = [0 0 0]; marker = 'x'; style = '-';
    end
end

