%% Verify convergence rate
% RPG:  log(y) = k * log(1 - beta/kappa) + c
%       1 - exp(slope) = beta/kappa
% 
% RAPG: log(y) = k * log(1 - beta/sqrt(kappa)) + c
%       1 - exp(slope) = beta/sqrt(kappa)

% seed = round(rand()*100000);
seed = 888;
fprintf('seed:%d\n', seed);
testnum = 20;
% data in RAPG
Slope = zeros(testnum, 1); % the slope of k ~ log(F(x_k)-F(x*))
Beta = zeros(testnum, 1); % 1 / (1 - exp(slope))
Kappa = zeros(testnum, 1); % condition number of Hessf(x*)
% data in RPG
slope = zeros(testnum, 1); % the slope of k ~ log(F(x_k)-F(x*))
beta = zeros(testnum, 1); % 1 / (1 - exp(slope))

for i = 1:testnum
    rng(seed);

    m = 20;
    n = 1000;
    p = 1;
    lambda = 1e-4;

    % Data matrix A
    tmp = linspace(0, 1, testnum);
    k = 10.^(2 * tmp.^2 - 2.0);
    noise = 0.00001;  
    [~,A,~] = Data_A(m, n, k(i), noise);
    
    [U, S, V] = svd(A, 'econ');
    PCAV = V(:, 1:p);
    Xinitial.main = PCAV + 0.01 * randn(n, p);
    Xinitial.main = orth(Xinitial.main);
    tol = 1e-10*n*p;

    % Compute fopt by RPN
    option.n = n; option.r = p; option.mu = lambda;
    option.tol = 1e-10*n*p;
    option.maxiter = 1000;
    option.x0 = Xinitial.main;
    option.stop = 1e-10;
    option.outputgap = 100;
    epsilon_set = 1e-2;
    option.epsilon = epsilon_set;
    [xopt, fopt] = driver_rpncg(A, option);
    fopt = fopt(end);
    [~, mins, maxs] = cond_hessf(xopt, A);
    
    % Parameter Settings
    xi = 1;  A0 =  0.001; 
    Ftol = - inf;
    LRPG = 5*maxs;
    Lu = LRPG;
    Ll = Lu;
    mu = mins;  rho = 0.002; 
    theta = max([(rho + (mu - rho)*xi) / Lu + 0.001, 1]);
    q = (mu - rho) / (theta * Lu - rho); 
    Kappa(i) = 1/q;

    fprintf('RPG\n');  maxiter = 10000;
    [xopt1, iter1, time1, fv1, nD1, sparsity1, avar1, fs1, neta1] = RPG(Xinitial, A, lambda, LRPG, tol, maxiter);
    fprintf('RAPG\n');  maxiter = 5000;
    [xopt2, iter2, time2, fv2, nD2, sparsity2, avar2, fs2, neta2] = RAPG(Xinitial, A, lambda, A0, xi, q, Ll, Lu, tol, maxiter, fv1);

    [Slope(i), yRAPG] = compute_slope(fs2 - fopt); % RAPG
    Beta(i) = 1 / (1 - exp(Slope(i)));
    [slope(i), yRPG] = compute_slope(fs1 - fopt); % RPG
    beta(i) = 1 / (1 - exp(slope(i)));
end

%%
xx = linspace(max(min(Kappa)-20,1), max(Kappa)+100, 100);
s1 = polyfit(sqrt(Kappa), Beta, 1);
s2 = polyfit(Kappa, beta, 1);
set(0,'defaultaxesfontsize',12, ...
   'defaultaxeslinewidth',0.7, ...
   'defaultlinelinewidth',.8,'defaultpatchlinewidth',0.8);
set(0,'defaultlinemarkersize',10)

figure('Position',[400 300 500 360]);
plot(Kappa, Beta,'*', "LineWidth",1);  hold on;
plot(xx, s1(2) + s1(1) * sqrt(xx), "LineWidth",1.8);  
% title('RAPG'); 
xlabel('$\kappa$','Interpreter','latex','FontSize',20);  
ylabel('$\frac{1}{1-e^s}$','Interpreter','latex','FontSize',20);

saveas(gcf, 'figure/rate_RAPG_1.eps', 'epsc')  
saveas(gcf, 'figure/rate_RAPG_1.pdf')  

figure('Position',[1000 300 500 360]);
plot(Kappa, beta,'*', "LineWidth",1);  hold on;
plot(xx, s2(2) + s2(1) * xx, "LineWidth",1.8);  
% title('RPG'); 
xlabel('$\kappa$','Interpreter','latex','FontSize',20);  
ylabel('$\frac{1}{1-e^s}$','Interpreter','latex','FontSize',20);

saveas(gcf, 'figure/rate_RPG_1.eps', 'epsc')  
saveas(gcf, 'figure/rate_RPG_1.pdf')
%% 
function [v,A,sparsity] = Data_A(m, n, k, noise)
% Generate the data matrix A
% m <= n
    v = randn(n, 1);
    for i = 1 : n
        if rand() <  0.50 
            v(i) = 0;
        else
            v(i) = v(i) + sign(v(i));
        end
    end
    sparsity = sum(abs(v) < 1e-8) / n;
    v = full(v);  v = v / norm(v);
    V = [v, null(v')];  
    S = [diag([m, m:-1:2]), zeros(m, n-m)];
    S(1,1) = m+k;
    U = randn(m, n);
    U = orth(U);
    A = U * S * V';
    A = A + noise * randn(m, n);
end

function [kappa, mins, maxs] = cond_hessf(x, A)
% Compute the Riemannian Hessian of f = - x^T A^T A x at x, x \in S^{n-1}
    n = length(x);
    Ax = A * x;
    H = -2 * A' * A + 2 * x * (A' * Ax)' + 2 * (Ax' * Ax) * eye(n);
    H = H - x * (x' * H);
    H = H - (H * x) * x';
    ss = real(eig(H));
    ss = sort(ss);
    maxs = ss(end); mins = ss(2);
    kappa = maxs / mins;
end

function [s, y] = compute_slope(y)
% Compute the slope s of k ~ log y(k) graph$
    y = log(y(y>1e-14));
    y = y(round(length(y)*0.8) : length(y));
    s = polyfit(1:length(y), y, 1);
    s = s(1);
end