
#include "RBFGSLPSub.h"

/*Define the namespace*/
namespace ROPTLIB{

	void RBFGSLPSub::Run(void)
	{
		DeleteVectors(W, Mani->GetIntrDim() + 1);
		NewVectors(W, Mani->GetIntrDim() + 1);
		Variable *xTemp;
		Vector *gfTemp;
		pre_funs.clear();

		Solvers::Run();

		/*If the linesearch condition Armijo-Glodstein condition, then the locking conidition is not necessary and don't output warning.
		If the pair of retraction and vector transport satisfies the locking condition, then output warning.
		If the idea in [Section 4.1, HGA2015] is used, then the locking condition is satisfied and don't output the warning.
		[HGA2015]: Wen Huang, K. A. Gallivan, and P.-A. Absil, A Broyden Class of Quais-Newton Methods for Riemannian Optimization,
		SIAM on Journal Optimization, (25)3, 1660-1685, 2015 */
		if (LineSearch_LS != ARMIJO && !Prob->GetDomain()->GetHasLockCon() && !Prob->GetDomain()->GetHasHHR() && Debug >= FINALRESULT)
		{
			std::cout << "Warning: The locking condition is not satisfied. Line search may fail!" << std::endl;
		}

		/*Choose the linesearch algorithm used in the algorithm*/

		if (LineSearch_LS != WOLFELP)
		{
			std::cout << "Only line search for Wolfe conidtions and Lipschitz continous function can be used!" << std::endl;
		}
		LSstatus = SUCCESS;
		f1 = Prob->f(x1); nf++;
		Prob->Grad(x1, gf1); ng++;
		ngf0 = sqrt(Mani->Metric(x1, gf1, gf1));
		ngf = ngf0;
		newslope = 0;
		iter = 0;
		if (Debug >= ITERRESULT)
		{
			printf("i:%d,f:%.3e,|gf|:%.3e,\n", iter, f1, ngf);
			timeSeries[iter] = static_cast<double>(getTickCount() - starttime) / CLK_PS;
			funSeries[iter] = f1;
			gradSeries[iter] = ngf;
		}
		bool isstop = false;

		/*Start the loop*/
		while ((((!isstop) && iter < Max_Iteration) || iter < Min_Iteration) && LSstatus == SUCCESS)
		{
			GetSearchDir(); // Obtain search direction eta1 and gf

			/*Call the function to check whether the stopping criterion is satisfied or not.
			The default function is written in Solvers.h and Solvers.cpp*/
			isstop = IsStopped();

			if (Pngf <= Del)
			{
				Eps *= Theta_eps;
				Eps = (Eps > Min_Eps) ? Eps : Min_Eps;
				Del *= Theta_del;
				if (Debug >= ITERRESULT)
					std::cout << "Shinking Epsilon and Delta to" << Eps << " and " << Del << " respectively." << std::endl;
				continue;
			}

			initialslope = -Pngf;
			/*Compute initial step size for the next iteration*/
			InitialStepSize();

			initiallength = stepsize;

			/* Call the specified linesearch algorithm.
			Note that in the linesearch algorithm, we need to obtain
			accepted stepsize, eta2=stepsize*eta1, x2 = R_{x_1}(eta_2), f2 = f(x2), and gf2 = grad f(x_2) */
			LinesearchWolfeLipschitz();

			/*Output debug information if necessary.*/
			if (LSstatus < SUCCESS && Debug >= FINALRESULT)
			{
				std::cout << "Linesearch fails! LSstatus:" << LSstatusSetnames[LSstatus] << std::endl;
			}

			iter++;

			/*Update the Hessian approximation for quasi-Newton methods
			or obtain search direction candadite for Riemannian nonlinear conjugate gradient*/
			UpdateData();

			/*norm of the gradient at x2*/
			ngf = sqrt(Mani->Metric(x2, gf2, gf2));

			if (Debug >= ITERRESULT)
			{
				/*Output information*/
				if (iter % OutputGap == 0)
				{
					PrintGenInfo();
					PrintInfo(); // Output information specific to Algorithms
				}
				/*Store debug information in the arrays*/
				timeSeries[iter] = static_cast<double>(getTickCount() - starttime) / CLK_PS;
				funSeries[iter] = f2; gradSeries[iter] = ngf;
			}

			/*Switch information at x1 and x2*/
			xTemp = x1; x1 = x2; x2 = xTemp;
			gfTemp = gf1; gf1 = gf2; gf2 = gfTemp;
			pre_funs.push_front(f1);
			if (pre_funs.size() > Num_pre_funs && pre_funs.size() > 1)
				pre_funs.pop_back();
			f1 = f2;
		}
		ComTime = static_cast<double>(getTickCount() - starttime) / CLK_PS;
		if (Debug >= ITERRESULT)
			lengthSeries = iter + 1;
		if (Debug >= FINALRESULT)
		{
			printf("Iter:%d,f:%.3e,|gf|:%.3e,|gf|/|gf0|:%.3e,time:%.2e,nf:%d,ng:%d,nR:%d,", iter, f2,
				ngf, ngf / ngf0, ComTime, nf, ng, nR);
			if (nH != 0)
			{
				printf("nH:%d,", nH);
			}
			if (nV != 0)
			{
				printf("nV(nVp):%d(%d),", nV, nVp);
			}
			printf("\n");
		}
	};

	bool RBFGSLPSub::IsStopped(void)
	{
		if (sqrt(Pngf) < Tolerance)
			return true;
		return false;
	};

	RBFGSLPSub::RBFGSLPSub(const Problem *prob, const Variable *initialx, LinearOPE *initialH)
	{
		Initialization(prob, initialx, initialH);
	};

	void RBFGSLPSub::Initialization(const Problem *prob, const Variable *initialx, LinearOPE *initialH)
	{
		SetProbX(prob, initialx, initialH);
		SetDefaultParams();
	};

	void RBFGSLPSub::SetProbX(const Problem *prob, const Variable *initialx, LinearOPE *initialH)
	{
		SolversLS::SetProbX(prob, initialx);

		const Vector *EMPTYETA;
		if (prob->GetDomain()->GetIsIntrinsic())
			EMPTYETA = prob->GetDomain()->GetEMPTYINTR();
		else
			EMPTYETA = prob->GetDomain()->GetEMPTYEXTR();
		bool initHisnull = (initialH == nullptr);
		if (initHisnull)
		{
			if (prob->GetDomain()->GetIsIntrinsic())
			{
				initialH = new LinearOPE(prob->GetDomain()->GetEMPTYINTR()->Getlength());
			}
			else
			{
				initialH = new LinearOPE(prob->GetDomain()->GetEMPTYEXTR()->Getlength());
			}
			initialH->ScaledIdOPE();
		}

		H = initialH->ConstructEmpty();
		tildeH = initialH->ConstructEmpty();
		initialH->CopyTo(H);
		s = EMPTYETA->ConstructEmpty();
		y = EMPTYETA->ConstructEmpty();
		v = EMPTYETA->ConstructEmpty();
		gf = EMPTYETA->ConstructEmpty();

		if (initHisnull)
			delete initialH;
		prob->SetUseGrad(true);
		prob->SetUseHess(false);
	};

	void RBFGSLPSub::SetDefaultParams()
	{
		SolversLS::SetDefaultParams();
		isconvex = false;
		InitSteptype = ONESTEP;
		Theta_eps = 0.1;
		Theta_del = 0.1;
		Eps = 0.01;
		Min_Eps = 1e-7;
		Del = 0.01;
		lambdaLower = 1e-7;
		lambdaUpper = 1e7;
		W = nullptr;
		SolversLS::SolverName.assign("RBFGSLPSub");
	};

	RBFGSLPSub::~RBFGSLPSub(void)
	{
		delete v;
		delete gf;
		delete s;
		delete y;
		delete H;
		delete tildeH;
		DeleteVectors(W, Mani->GetIntrDim() + 1);
	};

	void RBFGSLPSub::CheckParams(void)
	{
		SolversLS::CheckParams();
		char YES[] = "YES";
		char NO[] = "NO";
		char *status;

		std::cout << "RBFGSLPSub METHOD PARAMETERS:" << std::endl;
		status = (lambdaLower > 0 && lambdaLower < lambdaUpper) ? YES : NO;
		std::cout << "lambdaLower   :" << std::setw(15) << lambdaLower << "[" << status << "],\t";
		status = (lambdaUpper >= lambdaLower) ? YES : NO;
		std::cout << "lambdaUpper   :" << std::setw(15) << lambdaUpper << "[" << status << "]" << std::endl;
		status = (Eps > 0 && Eps < 1) ? YES : NO;
		std::cout << "Eps           :" << std::setw(15) << Eps << "[" << status << "],\t";
		status = (Theta_eps > 0 && Theta_eps < 1) ? YES : NO;
		std::cout << "Theta_eps     :" << std::setw(15) << Theta_eps << "[" << status << "]" << std::endl;
		status = (Min_Eps > 0 && Min_Eps < 1) ? YES : NO;
		std::cout << "Min_Eps       :" << std::setw(15) << Min_Eps << "[" << status << "],\t";
		status = (Del > 0 && Del < 1) ? YES : NO;
		std::cout << "Del           :" << std::setw(15) << Del << "[" << status << "]" << std::endl;
		status = (Theta_del > 0 && Theta_del < 1) ? YES : NO;
		std::cout << "Theta_del     :" << std::setw(15) << Theta_del << "[" << status << "],\t";
		status = YES;
		std::cout << "isconvex      :" << std::setw(15) << isconvex << "[" << status << "]" << std::endl;
	};

	void RBFGSLPSub::GetSearchDir(void)
	{
		gf1->CopyTo(W[0]);
		CurrentlengthW = 1;
		Sub_soln = new double[Mani->GetIntrDim() + 1];
		for (integer i = 0; i < Mani->GetIntrDim() + 1; i++)
			Sub_soln[i] = 1.0;

		for (integer i = 0; i < Mani->GetIntrDim(); i++)
		{
			/*gf = argmin \| v \|_P^2, v in convex hall of W, let eta1 = - P gf
			Pngf = \|gf\|_P^2. TOCHECK, variable gf seems not to be necessary.*/
			Pngf = MinPnormWv();
			if (Pngf <= Del)
			{
				delete[] Sub_soln;
				return;
			}
			neta1 = sqrt(Mani->Metric(x1, eta1, eta1));
			stepsize = Eps / neta1;
			f2 = h();
			hb = f2 - f1 + LS_alpha * Eps / neta1 * Pngf;
			if (hb <= 0)
			{
				delete[] Sub_soln;
				return;
			}
			/*get gf2 and stepsize*/
			Increasing();
			Mani->InverseVectorTransport(x1, eta2, x2, gf2, gf2);
			betay = Mani->Beta(x1, eta2);
			Mani->ScaleTimesVector(x1, 1.0 / betay, gf2, gf2);
			gf2->CopyTo(W[CurrentlengthW]);
			CurrentlengthW++;
			if (CurrentlengthW == Mani->GetIntrDim() + 1 && Debug >= ITERRESULT)
			{
				std::cout << "Warning: the number of W reaches its upper-bound!" << std::endl;
			}
		}
		delete[] Sub_soln;
	};

	void RBFGSLPSub::Increasing(void)
	{
		double a = 0, b = stepsize;
		integer times = 0;
		while (times < 10)
		{
			if (dh() + LS_alpha * Pngf < 0)
			{
				stepsize = (a + b) / 2;
				f2 = h();
				ht = f2 - f1 + LS_alpha * Eps / neta1 * Pngf;
				if (hb > ht)
				{
					a = stepsize;
				}
				else
				{
					b = stepsize;
				}
			}
			else
			{
				break;
			}
			times++;
		}
		if (times == 10 && Debug >= ITERRESULT)
			std::cout << "warning: the loop in RBFGSLPSub::Increasing reaches the upperbound!" << std::endl;
	};

	double RBFGSLPSub::MinPnormWv(void)
	{

		if (CurrentlengthW == 1)
		{
			W[0]->CopyTo(gf);
			HvRBFGSSub(W[0], eta1);
			const double *PWptr = eta1->ObtainReadData();
			const double *Wptr = W[0]->ObtainReadData();
			integer Length = eta1->Getlength();
			double result = ddot_(&Length, const_cast<double *> (Wptr), &GLOBAL::IONE, const_cast<double *> (PWptr), &GLOBAL::IONE);
			Mani->ScaleTimesVector(x1, -1.0, eta1, eta1);
			return result;
		}

		SphereConvexHull subprob(Mani, x1, W, CurrentlengthW, this, &QuasiNewton::HvRBFGSSub);
		Sphere Domain(CurrentlengthW);
		subprob.SetDomain(&Domain);
		SphereVariable InitX(CurrentlengthW);

		double *InitXptr = InitX.ObtainWriteEntireData();
		double r2 = sqrt(2.0);
		for (integer i = 0; i < InitX.Getlength(); i++)
			InitXptr[i] = Sub_soln[i] / r2;

		Solvers *LRBFGSsolver = new LRBFGS(&subprob, &InitX);
		LRBFGSsolver->Stop_Criterion = GRAD_F;
		LRBFGSsolver->Debug = NOOUTPUT;
		LRBFGSsolver->Run();
		double result = LRBFGSsolver->Getfinalfun();
		const Variable *soln = LRBFGSsolver->GetXopt();
		const double *solnptr = soln->ObtainReadData();
		for (integer i = 0; i < CurrentlengthW; i++)
			Sub_soln[i] = solnptr[i] * solnptr[i];
		const SharedSpace *SharedWxsq = soln->ObtainReadTempData("Wxsq");
		SharedWxsq->GetSharedElement()->CopyTo(gf);
		const SharedSpace *SharedPWxsq = soln->ObtainReadTempData("PWxsq");
		SharedPWxsq->GetSharedElement()->CopyTo(eta1);
		Mani->ScaleTimesVector(x1, -1.0, eta1, eta1);
		delete LRBFGSsolver;
		return result;
	};

	void RBFGSLPSub::UpdateData(void)
	{
		UpdateDataRBFGSSub();
	};

	void RBFGSLPSub::PrintInfo(void)
	{
		printf("\n\tbetay:%.3e,inpss:%.3e,inpsy:%.3e,IsUpdateHessian:%d,", betay, inpss, inpsy, isupdated);
		printf("\n");
	};

	void RBFGSLPSub::SetParams(PARAMSMAP params)
	{
		Solvers::SetParams(params);
		PARAMSMAP::iterator iter;
		for (iter = params.begin(); iter != params.end(); iter++)
		{
			if (iter->first == static_cast<std::string> ("Eps"))
			{
				Eps = ((static_cast<integer> (iter->second)) != 0);
			}
			else
			if (iter->first == static_cast<std::string> ("Theta_eps"))
			{
				Theta_eps = static_cast<integer> (iter->second);
			}
			else
			if (iter->first == static_cast<std::string> ("Min_Eps"))
			{
				Min_Eps = static_cast<integer> (iter->second);
			}
			else
			if (iter->first == static_cast<std::string> ("Del"))
			{
				Del = static_cast<integer> (iter->second);
			}
			else
			if (iter->first == static_cast<std::string> ("Theta_del"))
			{
				Theta_del = static_cast<integer> (iter->second);
			}
		}
	};
}; /*end of ROPTLIB namespace*/
