/*
This file defines the class of the line search algorithm for locally lipschitz functions on Riemannian manifolds

Solvers --> QuasiNewton --> SolversLS --> RBFGSLPSub

---- WH
*/

#ifndef RBFGSLPSUB_H
#define RBFGSLPSUB_H

#include "SolversLS.h"
#include "SphereConvexHull.h"
#include "Sphere.h"
#include "LRBFGS.h"
#include "def.h"

/*Define the namespace*/
namespace ROPTLIB{

	class RBFGSLPSub : public SolversLS{
	public:
		/*Run the algorithm. This function gives the framework for the linesearch method*/
		virtual void Run(void);

		/*The contructor of RBFGS method. It calls the function Solvers::Initialization.
		INPUT : prob is the problem which defines the cost function, gradient and possible the action of Hessian
		and specifies the manifold of domain.
		initialx is the initial iterate.
		initialH is the initial inverse Hessian approximation. If the input is nullptr, then the identity is used as the initial approximation*/
		RBFGSLPSub(const Problem *prob, const Variable *initialx, LinearOPE *initialH = nullptr);

		/*Destructor. Delete the vectors and Hessian approximation used in RBFGSLPSub, i.e., s and y, H and tildeH*/
		virtual ~RBFGSLPSub();

		/*PARAMSMAP is defined in "def.h" and it is a map from string to double, i.e., std::map<std::string, double> .
		This function is used to set the parameters by the mapping*/
		virtual void SetParams(PARAMSMAP params);

		/*Check whether the parameters about RBFGSLPSub are legal or not.*/
		virtual void CheckParams();

		/*Initialize the solvers by calling the "SetProbX" and "SetDefultParams" functions.
		INPUT:	prob is the problem which defines the cost function, gradient and possible the action of Hessian
		and specifies the manifold of domain.
		initialx is the initial iterate.
		initialH is the initial inverse Hessian approximation. If the input is nullptr, then the identity is used as the initial approximation*/
		virtual void Initialization(const Problem *prob, const Variable *initialx, LinearOPE *initialH = nullptr);

		/*Initialize the type of iterates x1, x2 and tangent vectors gf1, gf2, s, y, H and tildeH and obtian the problem and manifold information
		INPUT:	prob is the problem which defines the cost function, gradient and possible the action of Hessian
		and specifies the manifold of domain.
		initialx is the initial iterate.
		initialH is the initial inverse Hessian approximation. If the input is nullptr, then the identity is used as the initial approximation*/
		virtual void SetProbX(const Problem *prob, const Variable *initialx, LinearOPE *initialH = nullptr);

		/*Setting parameters (member variables) to be default values */
		virtual void SetDefaultParams();

		/*Check whether a stopping criterion is satisfied or not*/
		virtual bool IsStopped(void);

		/*Epsilon-subdifferential
		Default: 0.01*/
		double Eps;
		/*Reduce the Epsilon by Theta_eps each time, i.e., Eps *= Theta_eps. 
		Default: 0.1*/
		double Theta_eps;
		/*The minimum value of Eps. Too small epsilon have non-negligible numerical error.
		Default: 1e-7*/
		double Min_Eps;

		/*The bound on the square of P-norm of gradient 
		Default: 0.01*/
		double Del;
		/*Reduce Del by Theta_del each time.
		Default: 0.1*/
		double Theta_del;

	protected:
		double MinPnormWv(void);
		void Increasing(void);

		/*Compute the search direction */
		void GetSearchDir(void);

		/*Update the Hessian approximation if necessary*/
		void UpdateData(void);

		/*Print information specific to RBFGSLPSub*/
		virtual void PrintInfo();

		/*Algorithm-related variables*/
		Vector *v;
		Vector *gf;
		Vector **W;
		double *Sub_soln;

		integer CurrentlengthW; /*The current length of W*/
		double Pngf;
		double neta1;
		double ht, hb;
	};
}; /*end of ROPTLIB namespace*/
#endif // end of RBFGSLPSUB_H
