function fs = stiefel(retraction, vector_transport)
% this function return some necessary function handles of stiefel
% Input:
% Stiefel manifold : St(p, n)
%     params.retraction : 
%         '1', exponential mapping when it is considered as Quotient space
%         '2', exponential mapping when it is considered as Embedding space
%         '3', qf factorization : qf(X + eta)
%         '4', qf factorization : qf(X + eta)
%     params.vector_transport : the isometric vector transport
%         '1', noniso: vector transport by projection
%         '2', Emb Iso: parallel transport
%         '3', Emb Iso: direct-rotation with basis on tangent space
%         '4', Emb Iso: direct-rotation with basis on normal space
%         '5', Emb Iso: smooth basis on tangent space
%         '6', Emb Iso: smooth basis on normal space
%         '7', Emb Iso: intrinsic method
%         '8', Quo Iso: parallel transport
%         '9', Quo Iso: direct-rotation with basis on horizontal space
%         '10', Quo Iso: direct-rotation with basis on vertical space
%         '11',Quo Iso: smooth basis on horizontal space
%         '12',Quo Iso: smooth basis on vertical space
%         '13',Quo Iso: intrinsic method
% output:
%     fns.inpro(x, v1, v2) : return the inner product g_x(v1, v2) of two tangent vectors v1 and v2 on T_x M.
%     fns.Hv(x, H, v) : return tangent vector which is given by that operator H affect on tangent vector v on T_x M.
%     fns.invBv(x, B, v) : return a tangent vector vv on T_x M which satisfies B vv = v.
%     fns.phi() : return the coefficient of RBroyden Family update formula.
%     fns.proj(x, eta) : return P_x(eta) by projecting v to tangent space of x.
%     fns.SRTV(x, m) : return m random tangent vectors in T_x M.
%     fns.R(x, eta) : return R_x(eta), where R is a retraction, x is an element on the manifold and eta is a tangent vector of T_x M.
%     fns.Tranv(x1, d, x2, v) : return a tangent vector on T_x2 M, which is given by vector transport that transport v \in T_x M to T_{R_x(d)} M. Here x2 = R_x(d).
%     fns.invTranv(x1, d, x2, v) : return a tangent vector on T_x1 M, which is given by inverse vector transport that transport v \in T_x2 M to T_x1 M. Here x2 = R_x1(d).
%     fns.TranH(x1, d, x2, H1) : return a operator, H2, that affects on T_x2 M. H2 satisfies that for any v \in T_x2(M), H2(v) = Tran (H1 (Tran^{-1}(v)))
%                             where Tran is the vector transport fns.Tranv(x1, d, x2, v).
%     fns.rank1operator(x, v1, v2) : return a operator, H, on T_x M that satisfies that for any v \in T_x M, H(v) = g_x(v2, v) * v1.
%     fns.operadd(x, H1, H2) : return a operator, H, on T_x M that satisfies that for any v \in T_x M, H(v) = H1(v) + H2(v).

    fprintf('stiefel manifold\n');
    fs.Hv = @Hv;
    fs.invBv = @invBv;
    fs.phi = @phi;
    fs.operadd = @operadd;
    fs.proj = @proj;
    
    % using intrinsic approach
    if(vector_transport == 7 || vector_transport == 13 || vector_transport == 19 || vector_transport == 25)
        fprintf('intrinsic dimension approach\n')
        fs.Hv = @Hv_intr;
        fs.invBv = @invBv_intr;
        fs.proj = @proj_intr;
        fs.SRTV = @SRTV_intr;
        fs.rank1operator = @rank1operator_intr;
        fs.inpro = @inpro_intr;
        if(retraction == 1)
            fprintf('exponential mapping: stiefel manifold is considered as quotient space\n')
            fs.R = @R_exp_quotient_intr;
        elseif(retraction == 2)
            fprintf('exponential mapping: stiefel manifold is considered as embedded space\n')
            fs.R = @R_exp_embedded_intr;
        elseif(retraction == 3)
            fprintf('retraction: qf for stiefel manifold is considered as quotient space\n')
            fs.R = @R_qf_quotient_intr;
        elseif(retraction == 4)
            fprintf('retraction: qf for stiefel manifold is considered as embedded space\n')
            fs.R = @R_qf_embedded_intr;
        end
    else
        if(retraction == 1 || retraction == 3)
            fs.SRTV = @SRTV_quotient;
            fprintf('rank 1 operator: stiefel manifold is considered as quotient space\n')
            fs.rank1operator = @rank1operator_quotient;
            fprintf('inner product: stiefel manifold is considered as quotient space\n')
            fs.inpro = @inpro_quotient;
        else
            fs.SRTV = @SRTV_embedded;
            fprintf('rank 1 operator: stiefel manifold is considered as embedded space\n')
            fs.rank1operator = @rank1operator_embedded;
            fprintf('inner product: stiefel manifold is considered as embedded space\n')
            fs.inpro = @inpro_embedded;
        end

        if(retraction == 1)
            fprintf('exponential mapping: stiefel manifold is considered as quotient space\n')
            fs.R = @R_exp_quotient;
            fs.beta = @beta_exp;
        elseif(retraction == 2)
            fprintf('exponential mapping: stiefel manifold is considered as embedded space\n')
            fs.R = @R_exp_embedded;
            fs.beta = @beta_exp;
        elseif(retraction == 3)
            fprintf('retraction: qf for stiefel manifold is considered as quotient space\n')
            fs.R = @R_qf_quotient;
            fs.beta = @beta_qf_quotient;
        elseif(retraction == 4)
            fprintf('retraction: qf for stiefel manifold is considered as embedded space\n')
            fs.R = @R_qf_embedded;
            fs.beta = @beta_qf_embedded;
        end
    end
    
    if(vector_transport == 1)
        fprintf('Emb&Quo nonIso: vector transport by projection \n')
        fs.Tranv = @Tranv_proj;
        fs.Tranv_Params = @Tranv_proj;
        fs.invTranv = @invTranv_proj;
        fs.TranH = @TranH_proj;
    elseif(vector_transport == 2)
        fprintf('Emb Iso: parallel transport \n')
        fs.Tranv = @Tranv_parallel_embedded;
        fs.Tranv_Params = @Tranv_parallel_embedded;
        fs.invTranv = @invTranv_parallel_embedded;
        fs.TranH = @TranH_parallel_embedded;
    elseif(vector_transport == 3)
        fprintf('Emb Iso: direct-rotation with basis on tangent space \n')
        fs.Tranv = @Tranv_cano;
        fs.Tranv_Params = @Tranv_cano_Params;
        fs.invTranv = @invTranv_cano;
        fs.TranH = @TranH_cano;
    elseif(vector_transport == 4)
        fprintf('Emb Iso: direct-rotation with basis on normal space \n')
        fs.Tranv = @Tranv_cano_NonIn;
        fs.Tranv_Params = @Tranv_cano_NonIn_Params;
        fs.invTranv = @invTranv_cano_NonIn;
        fs.TranH = @TranH_cano_NonIn;
    elseif(vector_transport == 5)
        fprintf('Emb Iso: smooth basis on tangent space \n')
        fs.Tranv = @Tranv_smooth;
        fs.Tranv_Params = @Tranv_smooth_Params;
        fs.invTranv = @invTranv_smooth;
        fs.TranH = @TranH_smooth;
    elseif(vector_transport == 6)
        fprintf('Emb Iso: smooth basis on normal space \n')
        fs.Tranv = @Tranv_smooth_NonIn;
        fs.Tranv_Params = @Tranv_smooth_NonIn_Params;
        fs.invTranv = @invTranv_smooth_NonIn;
        fs.TranH = @TranH_smooth_NonIn;
    elseif(vector_transport == 7)
        fprintf('Emb Iso: intrinsic method \n')
        fs.Tranv = @Tranv_smooth_intr;
        fs.Tranv_Params = @Tranv_smooth_intr_Params;
        fs.invTranv = @invTranv_smooth_intr;
        fs.TranH = @TranH_smooth_intr;
    elseif(vector_transport == 8)
        fprintf('Quo Iso: parallel transport \n')
        fs.Tranv = @Tranv_parallel_quotient;
        fs.Tranv_Params = @Tranv_parallel_quotient;
        fs.invTranv = @invTranv_parallel_quotient;
        fs.TranH = @TranH_parallel_quotient;
    elseif(vector_transport == 9)
        fprintf('Quo Iso: direct-rotation with basis on horizontal space \n')
        fs.Tranv = @Tranv_cano_Quo;
        fs.Tranv_Params = @Tranv_cano_Quo_Params;
        fs.invTranv = @invTranv_cano_Quo;
        fs.TranH = @TranH_cano_Quo;
    elseif(vector_transport == 10)
        fprintf('Quo Iso: direct-rotation with basis on vertical space \n')
        fs.Tranv = @Tranv_cano_Quo_NonIn;
        fs.Tranv_Params = @Tranv_cano_Quo_NonIn_Params;
        fs.invTranv = @invTranv_cano_Quo_NonIn;
        fs.TranH = @TranH_cano_Quo_NonIn;
    elseif(vector_transport == 11)
        fprintf('Quo Iso: smooth basis on horizontal space \n')
        fs.Tranv = @Tranv_smooth_Quo;
        fs.Tranv_Params = @Tranv_smooth_Quo_Params;
        fs.invTranv = @invTranv_smooth_Quo;
        fs.TranH = @TranH_smooth_Quo;
    elseif(vector_transport == 12)
        fprintf('Quo Iso: smooth basis on vertical space \n')
        fs.Tranv = @Tranv_smooth_Quo_NonIn;
        fs.Tranv_Params = @Tranv_smooth_Quo_NonIn_Params;
        fs.invTranv = @invTranv_smooth_Quo_NonIn;
        fs.TranH = @TranH_smooth_Quo_NonIn;
    elseif(vector_transport == 13)
        fprintf('Quo Iso: intrinsic method \n')
        fs.Tranv = @Tranv_smooth_Quo_intr;
        fs.Tranv_Params = @Tranv_smooth_Quo_intr_Params;
        fs.invTranv = @invTranv_smooth_Quo_intr;
        fs.TranH = @TranH_smooth_Quo_intr;
    end
end

function output = Hv(x, H, v)
    [n, p] = size(x);
    v = reshape(v, n * p, 1);
    output = reshape(H * v, n, p);
end

function output = Hv_intr(x, H, v)
    output = H * v;
end

function output = invBv(x, B, v)
    [n, p] = size(x);
    ind = 1;
    v = reshape(v, n * p, 1);
    for i = 1 : p
        for j = 1 : (p - i + 1)
            B(n * p + ind, (i - 1) * n + 1 : i * n) = x(:, j + i - 1)';
            B(n * p + ind, (i - 1 + j - 1) * n + 1 : (i + j - 1) * n) = x(:, i)';
            ind = ind + 1;
        end
    end
    v(n * p + 1 : n * p + p * (p + 1) / 2) = 0;
    output = reshape(linsolve(B, v), n, p);
end

function output = invBv_intr(x, B, v)
    output = linsolve(B, v);
end

function output = phi()
    output = 0;
end

function output = SRTV_quotient(x, m, seed)
    rand('state', seed);
    basis_x = build_bases_quotient(x);
    [nn, d] = size(basis_x);
    [n, p] = size(x);
    u = rand(d, m) - 0.5;
    TV = basis_x * u;
    for i = 1 : m
        output{i} = reshape(TV(:, i), n, p);
    end
end

function output = SRTV_embedded(x, m, seed)
    rand('state', seed);
    basis_x = build_bases_embedded(x);
    [nn, d] = size(basis_x);
    [n, p] = size(x);
    u = rand(d, m) - 0.5;
    TV = basis_x * u;
    for i = 1 : m
        output{i} = reshape(TV(:, i), n, p);
    end
end

function output = SRTV_intr(x, m, seed)
    rand('state', seed);
    [n, p] = size(x{1});
    d = 0.5 * p * (p - 1) + (n - p) * p;
    u = rand(d, m) - 0.5;
    for i = 1 : m
        output{i} = u(:, i);
    end
end

function output = proj(x, eta)
    temp = x' * eta;
    output = eta - x * ((temp + temp') / 2);
end

function output = proj_intr(x, eta)
    output = eta;
end

function output = rank1operator_embedded(x, v1, v2)
    [n, p] = size(x);
    v1 = reshape(v1, n * p, 1);
    v2 = reshape(v2, n * p, 1);
    output = v1 * v2';
end

function output = rank1operator_quotient(x, v1, v2)
    [n, p] = size(x);
    v1 = reshape(v1, n * p, 1);
    v2 = reshape(v2 - 0.5 * x * (x' * v2), n * p, 1);
    output = v1 * v2';
end

function output = rank1operator_intr(x, v1, v2)
    output = v1 * v2';
end

function output = operadd(x, H1, H2)
    output = H1 + H2;
end

function output = inpro_embedded(x, v1, v2)
    output = sum(sum(v1 .* v2));
end

function output = inpro_quotient(x, v1, v2)
    v1 = reshape(v1, size(x));
    v2 = reshape(v2, size(x));
    output = sum(sum(v1 .* (v2 - 0.5 * x * (x' * v2))));
end

function output = inpro_intr(x, v1, v2)
    output = v1' * v2;
end

%% exponential mapping when it is considered as an embedded manifold
function output = R_exp_embedded(x, eta)
    [n, p] = size(x);
    A = x' * eta;
    S = eta' * eta;
    temp = expm([A -S; eye(p) A]) * eye(2 * p, p);
    output = [x eta] * temp * expm(- A);
%     %The following step is not needed analytically. However, it is needed
%     %numerically.
%     output = qf(output);
end

function output = R_exp_embedded_intr(x, eta)
    [n, p] = size(x);
    basisx = build_bases_embedded(x);
    eta = basisx * eta;
    eta = reshape(eta, n, p);
    A = x' * eta;
    S = eta' * eta;
    temp = expm([A -S; eye(p) A]) * eye(2 * p, p);
    output = [x eta] * temp * expm(- A);
    output = basisx' * reshape(output, [], 1);
end

%% exponential mapping when it is considered as an quotient manifold
function output = R_exp_quotient(x, eta)
    [n, p] = size(x);
    A = x' * eta;
    [Q, B] = qr(eta - x * (x' * eta), 0);
    Temp = expm([A -B'; B zeros(p, p)]);
    output = [x Q] * Temp(:, 1 : p);
%     %The following step is not needed analytically. However, it is needed
%     %numerically.
%     output = qf(output);
end

function output = R_exp_quotient_intr(x, eta)
    [n, p] = size(x);
    basisx = build_bases_quotient(x);
    basisx_flat = GB_quotient(x, basisx)';
    eta = basisx * eta;
    eta = reshape(eta, n, p);
    A = x' * eta;
    [Q, B] = qr(eta - x * (x' * eta), 0);
    Temp = expm([A -B'; B zeros(p, p)]);
    output = [x Q] * Temp(:, 1 : p);
    output = basisx_flat * reshape(output, [], 1);
end

%% qf
function output = R_qf_embedded(x, eta)
    output = qf(x + eta);
end

function output = R_qf_embedded_intr(x, eta)
    eta = embedded_full_c(x, eta);
    output{1} = qf(x{1} + eta);
    output{2} = null(output{1}');
end

function output = R_qf_quotient(x, eta)
    output = qf(x + eta);
end

function output = R_qf_quotient_intr(x, eta)
    eta = quotient_full_c(x, eta);
    output{1} = qf(x{1} + eta);
    output{2} = null(output{1}');
end

function output = Tranv_R_qf(x, d, y, v)
    temp = inv(y' * (x + d));
    temp2 = v * temp;
    temp3 = y' * temp2;
    output = y * (rho_skew(temp3) - temp3) + temp2;
%     output = y * rho_skew(temp3) + temp2 - y * temp3;%%---------------
end

function output = rho_skew(A)
    output = tril(A, -1) - tril(A, -1)';
end

%% polar
function output = R_polar(x, eta)
    [n, p] = size(x);
    output = (x + eta) * (eye(p) + eta' * eta)^(-0.5);
end

%% Emb&Quo nonIso: vector transport by projection
function [output, TranvParams] = Tranv_proj(x, d, y, v, TranvParams)
    temp = y' * v;
    output = v - y * ((temp + temp') / 2);
%     output = v - y * (y' * v + v' * y) / 2;%%-------
%     [n, p] = size(x);
%     perp_basis_x = build_perp_bases(x);
%     perp_basis_y = build_perp_bases(y);
%     output = reshape(v, n * p, 1) - perp_basis_x * (pinv(perp_basis_y' * perp_basis_x) * (perp_basis_y' * reshape(v, n * p, 1)));
%     output = reshape(output, n, p);
%     TranvParams{1} = perp_basis_y;
%     TranvParams{2} = perp_basis_x;
end

function [output, TranvParams] = invTranv_proj(x, d, y, v, TranvParams)
    [n, p] = size(x);
    perp_basis_x = build_perp_bases(x);
    perp_basis_y = build_perp_bases(y);
    output = reshape(v, n * p, 1) - perp_basis_y * (pinv(perp_basis_x' * perp_basis_y) * (perp_basis_x' * reshape(v, n * p, 1)));
    output = reshape(output, n, p);
    TranvParams{1} = perp_basis_y;
    TranvParams{2} = perp_basis_x;
    
%     output = v - x * (x' * v + v' * x) / 2;
end

function output = TranH_proj(x, d, y, H)
    [n, p] = size(x);
    perp_basis_x = build_perp_bases(x);
    perp_basis_y = build_perp_bases(y);
    output = H - H * perp_basis_y * pinv(perp_basis_x' * perp_basis_y) * perp_basis_x';
    output = output - perp_basis_y * (perp_basis_y' * output);
%     output = H - H * perp_basis_x * perp_basis_x';
%     output = output - perp_basis_x * pinv(perp_basis_y' * perp_basis_x) * (perp_basis_y' * output);
end

%% Emb Iso: parallel transport
function [output, TranvParams] = Tranv_parallel_embedded(x, d, y, v, TranvParams)
    [n, p] = size(x);
    A = x' * d;
    S = d' * d;
    M1 = [A -S; eye(p), A];
    M2 = [x d];
    fhandle = @(t, w)ode_parallel_vt_embedded(t, w, M1, M2, A, n, p);
%     options = odeset('RelTol',1e-5, 'AbsTol',1e-5);
%     [T,V] = ode45(fhandle, [0 1], reshape(v, n * p, 1), options);
%     output = reshape(V(end, :), n, p);
    [T, V] = runge_kutta1(fhandle, reshape(v, n * p, 1), 0.1, 0, 1);
    output = reshape(V(:, end), n, p);
    TranvParams = [];
end

function [output, TranvParams] = invTranv_parallel_embedded(x, d, y, v, TranvParams)
    invd = - Tranv_parallel_embedded(x, d, y, d);
    [output, TranvParams] = Tranv_parallel_embedded(y, invd, x, v);
end

function output = TranH_parallel_embedded(x, d, y, H)
    [n, p] = size(x);
    dim = n * p - p * (p + 1) / 2;
    bases = full(build_bases_embedded(x));
    P_bases = zeros(n * p, dim);
    P_Hb = zeros(n * p, dim);
    for i = 1 : dim
        P_bases(:, i) = reshape(Tranv_parallel_embedded(x, d, y, reshape(bases(:, i), n, p)), n * p, 1);
    end
    Hb = H * bases;
    for i = 1 : dim
        P_Hb(:, i) = reshape(Tranv_parallel_embedded(x, d, y, reshape(Hb(:, i), n, p)), n * p, 1);
    end
    output = P_Hb * P_bases';
end

%% Emb Iso: direct-rotation with basis on tangent space
function [output, TranvParams] = Tranv_cano(x, d, y, v, TranvParams)
    [n, p] = size(v);
    xbases = build_bases_embedded(x);
    ybases = build_bases_embedded(y);
    [U, S, V] = svd(full(ybases' * xbases));
    v = reshape(v, n * p, 1);
    output = reshape(ybases * (U * (V' * (xbases' * v))), n, p);
    TranvParams{1} = xbases;
    TranvParams{2} = ybases;
    TranvParams{3} = U;
    TranvParams{4} = V;
end

function [output, TranvParams] = Tranv_cano_Params(x, d, y, v, TranvParams)
    xbases = TranvParams{1};
    ybases = TranvParams{2};
    U = TranvParams{3};
    V = TranvParams{4};
    [n, p] = size(v);
    v = reshape(v, n * p, 1);
    output = reshape(ybases * (U * (V' * (xbases' * v))), n, p);
end

function [output, TranvParams] = invTranv_cano(x, d, y, v, TranvParams)
    [n, p] = size(v);
    xbases = build_bases_embedded(x);
    ybases = build_bases_embedded(y);
    [U, S, V] = svd(full(ybases' * xbases));
    v = reshape(v, n * p, 1);
    output = reshape(xbases * (V * (U' * (ybases' * v))), n, p);
    TranvParams{1} = xbases;
    TranvParams{2} = ybases;
    TranvParams{3} = U;
    TranvParams{4} = V;
end

function output = TranH_cano(x, d, y, H)
    xbases = build_bases_embedded(x);
    ybases = build_bases_embedded(y);
    [U, S, V] = svd(full(ybases' * xbases));
    temp = U * V';
    output = ybases * (temp * (xbases' * (H * xbases)) * temp') * ybases';
end

%% Emb Iso: direct-rotation with basis on normal space
function [output, TranvParams] = Tranv_cano_NonIn(x, d, y, v, TranvParams)
    [n, p] = size(x);
    v = reshape(v, n * p, 1);
    Bx_perp = build_perp_bases_embedded(x);
    By_perp = build_perp_bases_embedded(y);
    
    Q = qf_thin(full(By_perp - Bx_perp * (Bx_perp' * By_perp)));
    tilde_Q = qf_thin(full(Bx_perp - By_perp * (By_perp' * Bx_perp)));
    [U, S, V] = svd(tilde_Q' * Q);
    output = reshape((v + (tilde_Q * (U * V') - Q) * (Q' * v)), n, p);
    TranvParams{1} = Q;
    TranvParams{2} = tilde_Q;
    TranvParams{3} = U;
    TranvParams{4} = V;
end

function [output, TranvParams] = Tranv_cano_NonIn_Params(x, d, y, v, TranvParams)
    Q = TranvParams{1};
    tilde_Q = TranvParams{2};
    U = TranvParams{3};
    V = TranvParams{4};
    [n, p] = size(x);
    v = reshape(v, n * p, 1);
    output = reshape((v + (tilde_Q * (U * V') - Q) * (Q' * v)), n, p);
end

function [output, TranvParams] = invTranv_cano_NonIn(x, d, y, v, TranvParams)
    [n, p] = size(x);
    v = reshape(v, n * p, 1);
    Bx_perp = build_perp_bases_embedded(x);
    By_perp = build_perp_bases_embedded(y);
    Q = qf_thin(full(By_perp - Bx_perp * (Bx_perp' * By_perp)));
    tilde_Q = qf_thin(full(Bx_perp - By_perp * (By_perp' * Bx_perp)));
    [U, S, V] = svd(tilde_Q' * Q);
    output = reshape((v + (Q * (V * U') - tilde_Q) * (tilde_Q' * v)), n, p);
    TranvParams{1} = Q;
    TranvParams{2} = tilde_Q;
    TranvParams{3} = U;
    TranvParams{4} = V;
end

function output = TranH_cano_NonIn(x, d, y, H)
    [n, p] = size(x);
    Bx_perp = build_perp_bases_embedded(x);
    By_perp = build_perp_bases_embedded(y);
    Q = qf_thin(full(By_perp - Bx_perp * (Bx_perp' * By_perp)));
    tilde_Q = qf_thin(full(Bx_perp - By_perp * (By_perp' * Bx_perp)));
    [U, S, V] = svd(tilde_Q' * Q);
    output = H + H * (Q * (V * U') - tilde_Q) * tilde_Q';
    output = output + (tilde_Q * (U * V') - Q) * (Q' * output);
end

%% Emb Iso: smooth basis on tangent space
function [output, TranvParams] = Tranv_smooth(x, d, y, v, TranvParams)
    [n, p] = size(v);
    xbases = build_bases_embedded(x);
    ybases = build_bases_embedded(y);
    v = reshape(v, n * p, 1);
    output = reshape(ybases * (xbases' * v), n, p);
    TranvParams{1} = xbases;
    TranvParams{2} = ybases;
end

function [output, TranvParams] = Tranv_smooth_Params(x, d, y, v, TranvParams)
    xbases = TranvParams{1};
    ybases = TranvParams{2};
    [n, p] = size(v);
    v = reshape(v, n * p, 1);
    output = reshape(ybases * (xbases' * v), n, p);
end

function [output, TranvParams] = invTranv_smooth(x, d, y, v, TranvParams)
    [n, p] = size(v);
    xbases = build_bases_embedded(x);
    ybases = build_bases_embedded(y);
    v = reshape(v, n * p, 1);
    output = reshape(xbases * (ybases' * v), n, p);
    TranvParams{1} = xbases;
    TranvParams{2} = ybases;
end

function output = TranH_smooth(x, d, y, H)
    xbases = build_bases_embedded(x);
    ybases = build_bases_embedded(y);
    output = ybases * (xbases' * (H * xbases)) * ybases';
end

%% Emb Iso: smooth basis on normal space
function [output, TranvParams] = Tranv_smooth_NonIn(x, d, y, v, TranvParams)
    [n, p] = size(x);
    v = reshape(v, n * p, 1);
    Bx_perp = build_perp_bases_embedded(x);
    By_perp = build_perp_bases_embedded(y);
    Q = qf_thin(full(By_perp - Bx_perp * (Bx_perp' * By_perp)));
    tilde_Q = qf_thin(full(Bx_perp - By_perp * (By_perp' * Bx_perp)));
    output = reshape((v - (tilde_Q + Q) * (Q' * v)), n, p);
    TranvParams{1} = Q;
    TranvParams{2} = tilde_Q;
end

function [output, TranvParams] = Tranv_smooth_NonIn_Params(x, d, y, v, TranvParams)
    Q = TranvParams{1};
    tilde_Q = TranvParams{2};
    [n, p] = size(x);
    v = reshape(v, n * p, 1);
    output = reshape((v - (tilde_Q + Q) * (Q' * v)), n, p);
end

function [output, TranvParams] = invTranv_smooth_NonIn(x, d, y, v, TranvParams)
    [n, p] = size(x);
    v = reshape(v, n * p, 1);
    Bx_perp = build_perp_bases_embedded(x);
    By_perp = build_perp_bases_embedded(y);
    Q = qf_thin(full(By_perp - Bx_perp * (Bx_perp' * By_perp)));
    tilde_Q = qf_thin(full(Bx_perp - By_perp * (By_perp' * Bx_perp)));
    output = reshape((v - (Q + tilde_Q) * (tilde_Q' * v)), n, p);
    TranvParams{1} = Q;
    TranvParams{2} = tilde_Q;
end

function output = TranH_smooth_NonIn(x, d, y, H)
    [n, p] = size(x);
    Bx_perp = build_perp_bases_embedded(x);
    By_perp = build_perp_bases_embedded(y);
    Q = qf_thin(full(By_perp - Bx_perp * (Bx_perp' * By_perp)));
    tilde_Q = qf_thin(full(Bx_perp - By_perp * (By_perp' * Bx_perp)));
    output = H - H * (Q + tilde_Q) * tilde_Q';
    output = output - (tilde_Q + Q) * (Q' * output);
end

%% Emb Iso: intrinsic method
function [output, TranvParams] = Tranv_smooth_intr(x, d, y, v, TranvParams)
    TranvParams = [];
    output = v;
end

function [output, TranvParams] = Tranv_smooth_intr_Params(x, d, y, v, TranvParams)
    output = v;
end

function [output, TranvParams] = invTranv_smooth_intr(x, d, y, v, TranvParams)
    TranvParams = [];
    output = v;
end

function output = TranH_smooth_intr(x, d, y, H)
    output = H;
end

%% Quo Iso: parallel transport
function [output, TranvParams] = Tranv_parallel_quotient(x, d, y, v, TranvParams)
    [n, p] = size(x);
    A = x' * d;
    [Q, B] = qr(d - x * (x' * d), 0);
    M1 = [A -B'; B zeros(p, p)];
    M2 = [x Q];
    fhandle = @(t, w)ode_parallel_vt_quotient(t, w, M1, M2, A, n, p);
%     options = odeset('RelTol',1e-5, 'AbsTol',1e-5);
%     [T,V] = ode45(fhandle, [0 1], reshape(v, n * p, 1), options);
%     output = reshape(V(end, :), n, p);
    [T, V] = runge_kutta1(fhandle, reshape(v, n * p, 1), 0.1, 0, 1);
    output = reshape(V(:, end), n, p);
    TranvParams = [];
end

function [output, TranvParams] = invTranv_parallel_quotient(x, d, y, v, TranvParams)
    invd = - Tranv_parallel_quotient(x, d, y, d);
    [output, TranvParams] = Tranv_parallel_quotient(y, invd, x, v);
end

function output = TranH_parallel_quotient(x, d, y, H)
    [n, p] = size(x);
    dim = n * p - p * (p + 1) / 2;
    bases = full(build_bases_quotient(x));
    P_bases = zeros(n * p, dim);
    P_Hb = zeros(n * p, dim);
    for i = 1 : dim
        P_bases(:, i) = reshape(Tranv_parallel_quotient(x, d, y, reshape(bases(:, i), n, p)), n * p, 1);
    end
    Hb = H * bases;
    for i = 1 : dim
        P_Hb(:, i) = reshape(Tranv_parallel_quotient(x, d, y, reshape(Hb(:, i), n, p)), n * p, 1);
    end
    for i = 1 : p
        P_bases((i - 1) * n + 1 : i * n, :) = P_bases((i - 1) * n + 1 : i * n, :) - 0.5 * y * y' * P_bases((i - 1) * n + 1 : i * n, :);
    end
    output = P_Hb * P_bases';
end

%% Quo Iso: direct-rotation with basis on horizontal space
function [output, TranvParams] = Tranv_cano_Quo(x, d, y, v, TranvParams)
    [n, p] = size(v);
    xbases = build_bases_quotient(x);
    ybases = build_bases_quotient(y);
    dim = p * (n - (p + 1) / 2);
    xbases_flat = (xbases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    ybases_flat = (ybases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    [U, S, V] = svd(full(ybases_flat * xbases));
    v = reshape(v, n * p, 1);
    output = reshape(ybases * (U * (V' * (xbases_flat * v))), n, p);
    TranvParams{1} = xbases;
    TranvParams{2} = ybases;
    TranvParams{3} = U;
    TranvParams{4} = V;
end

function [output, TranvParams] = Tranv_cano_Quo_Params(x, d, y, v, TranvParams)
    xbases = TranvParams{1};
    ybases = TranvParams{2};
    U = TranvParams{3};
    V = TranvParams{4};
    [n, p] = size(v);
    dim = p * (n - (p + 1) / 2);
    xbases_flat = (xbases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    v = reshape(v, n * p, 1);
    output = reshape(ybases * (U * (V' * (xbases_flat * v))), n, p);
end

function [output, TranvParams] = invTranv_cano_Quo(x, d, y, v, TranvParams)
    [n, p] = size(v);
    xbases = build_bases_quotient(x);
    ybases = build_bases_quotient(y);
    dim = p * (n - (p + 1) / 2);
    ybases_flat = (ybases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    [U, S, V] = svd(full(ybases_flat * xbases));
    v = reshape(v, n * p, 1);
    output = reshape(xbases * (V * (U' * (ybases_flat * v))), n, p);
    TranvParams{1} = xbases;
    TranvParams{2} = ybases;
    TranvParams{3} = U;
    TranvParams{4} = V;
end

function output = TranH_cano_Quo(x, d, y, H)
    [n, p] = size(d);
    xbases = build_bases_quotient(x);
    ybases = build_bases_quotient(y);
    dim = p * (n - (p + 1) / 2);
    ybases_flat = (ybases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    xbases_flat = (xbases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    [U, S, V] = svd(full(ybases_flat * xbases));
    temp = U * V';
    output = ybases * (temp * (xbases_flat * (H * xbases)) * temp') * ybases_flat;
end

%% Quo Iso: direct-rotation with basis on vertical space
function [output, TranvParams] = Tranv_cano_Quo_NonIn(x, d, y, v, TranvParams)
    [n, p] = size(x);
    Bx_perpG05 = build_perp_bases_embedded(x);
    By_perpG05 = build_perp_bases_embedded(y);
    Qx = qf_thin(full(By_perpG05 - Bx_perpG05 * (Bx_perpG05' * By_perpG05)));
    Qy = qf_thin(full(Bx_perpG05 - By_perpG05 * (By_perpG05' * Bx_perpG05)));
    [U, S, V] = svd(Qy' * Qx);
    v = reshape(v - (1 - 1 / sqrt(2)) * x * (x' * v), n * p, 1);
    output = reshape((v + (Qy * (U * V') - Qx) * (Qx' * v)), n, p);
    output = output + (sqrt(2) - 1) * y * (y' * output);
    TranvParams{1} = Qx;
    TranvParams{2} = Qy;
    TranvParams{3} = U;
    TranvParams{4} = V;
end

function [output, TranvParams] = Tranv_cano_Quo_NonIn_Params(x, d, y, v, TranvParams)
    Qx = TranvParams{1};
    Qy = TranvParams{2};
    U = TranvParams{3};
    V = TranvParams{4};
    [n, p] = size(x);
    v = reshape(v - (1 - 1 / sqrt(2)) * x * (x' * v), n * p, 1);
    output = reshape((v + (Qy * (U * V') - Qx) * (Qx' * v)), n, p);
    output = output + (sqrt(2) - 1) * y * (y' * output);
end

function [output, TranvParams] = invTranv_cano_Quo_NonIn(x, d, y, v, TranvParams)
    [n, p] = size(x);
    Bx_perpG05 = build_perp_bases_embedded(x);
    By_perpG05 = build_perp_bases_embedded(y);
    Qx = qf_thin(full(By_perpG05 - Bx_perpG05 * (Bx_perpG05' * By_perpG05)));
    Qy = qf_thin(full(Bx_perpG05 - By_perpG05 * (By_perpG05' * Bx_perpG05)));
    [U, S, V] = svd(Qy' * Qx);
    v = reshape(v - (1 - 1 / sqrt(2)) * y * (y' * v), n * p, 1);
    output = (v + (Qx * (V * U') - Qy) * (Qy' * v));
    output = reshape(output, n, p);
    output = output + (sqrt(2) - 1) * x * (x' * output);
    TranvParams{1} = Qx;
    TranvParams{2} = Qy;
    TranvParams{3} = U;
    TranvParams{4} = V;
end

function output = TranH_cano_Quo_NonIn(x, d, y, H)
    [n, p] = size(x);
    Bx_perpG05 = build_perp_bases_embedded(x);
    By_perpG05 = build_perp_bases_embedded(y);
    Qx = qf_thin(full(By_perpG05 - Bx_perpG05 * (Bx_perpG05' * By_perpG05)));
    Qy = qf_thin(full(Bx_perpG05 - By_perpG05 * (By_perpG05' * Bx_perpG05)));
    [U, S, V] = svd(Qy' * Qx);
    output = BGn05_quotient(x, H);
    output = G05B_quotient(x, output);
    output = output + output * (Qx * (V * U') - Qy) * Qy';
    output = output + (Qy * (U * V') - Qx) * (Qx' * output);
    output = BG05_quotient(y, output);
    output = Gn05B_quotient(y, output);
end

%% Quo Iso: smooth basis on horizontal space
function [output, TranvParams] = Tranv_smooth_Quo(x, d, y, v, TranvParams)
    [n, p] = size(v);
    xbases = build_bases_quotient(x);
    ybases = build_bases_quotient(y);
    dim = p * (n - (p + 1) / 2);
    xbases_flat = (xbases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    v = reshape(v, n * p, 1);
    output = reshape(full(ybases * (xbases_flat * v)), n, p);
    TranvParams{1} = xbases;
    TranvParams{2} = ybases;
end

function [output, TranvParams] = Tranv_smooth_Quo_Params(x, d, y, v, TranvParams)
    xbases = TranvParams{1};
    ybases = TranvParams{2};
    [n, p] = size(v);
    dim = p * (n - (p + 1) / 2);
    xbases_flat = (xbases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    v = reshape(v, n * p, 1);
    output = reshape(full(ybases * (xbases_flat * v)), n, p);
end

function [output, TranvParams] = invTranv_smooth_Quo(x, d, y, v, TranvParams)
    [n, p] = size(v);
    xbases = build_bases_quotient(x);
    ybases = build_bases_quotient(y);
    dim = p * (n - (p + 1) / 2);
    ybases_flat = (ybases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    v = reshape(v, n * p, 1);
    output = reshape(full(xbases * (ybases_flat * v)), n, p);
    TranvParams{1} = xbases;
    TranvParams{2} = ybases;
end

function output = TranH_smooth_Quo(x, d, y, H)
    [n, p] = size(d);
    xbases = build_bases_quotient(x);
    ybases = build_bases_quotient(y);
    dim = p * (n - (p + 1) / 2);
    xbases_flat = (xbases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    ybases_flat = (ybases * spdiags([ones(p * (p - 1) / 2, 1) / 2; ones((n - p)* p, 1)], 0, dim, dim))';
    output = ybases * (xbases_flat * (H * xbases)) * ybases_flat;
end

%% Quo Iso: smooth basis on vertical space
function [output, TranvParams] = Tranv_smooth_Quo_NonIn(x, d, y, v, TranvParams)
    [n, p] = size(x);
    Bx_perpG05 = build_perp_bases_embedded(x);
    By_perpG05 = build_perp_bases_embedded(y);
    Qx = qf_thin(full(By_perpG05 - Bx_perpG05 * (Bx_perpG05' * By_perpG05)));
    Qy = qf_thin(full(Bx_perpG05 - By_perpG05 * (By_perpG05' * Bx_perpG05)));
    v = reshape(v - (1 - 1 / sqrt(2)) * x * (x' * v), n * p, 1);
    output = reshape((v - (Qy + Qx) * (Qx' * v)), n, p);
    output = output + (sqrt(2) - 1) * y * (y' * output);
    TranvParams{1} = Qx;
    TranvParams{2} = Qy;
end

function [output, TranvParams] = Tranv_smooth_Quo_NonIn_Params(x, d, y, v, TranvParams)
    Qx = TranvParams{1};
    Qy = TranvParams{2};
    [n, p] = size(x);
    v = reshape(v - (1 - 1 / sqrt(2)) * x * (x' * v), n * p, 1);
    output = reshape((v - (Qy + Qx) * (Qx' * v)), n, p);
    output = output + (sqrt(2) - 1) * y * (y' * output);
end

function [output, TranvParams] = invTranv_smooth_Quo_NonIn(x, d, y, v, TranvParams)
    [n, p] = size(x);
    Bx_perpG05 = build_perp_bases_embedded(x);
    By_perpG05 = build_perp_bases_embedded(y);
    Qx = qf_thin(full(By_perpG05 - Bx_perpG05 * (Bx_perpG05' * By_perpG05)));
    Qy = qf_thin(full(Bx_perpG05 - By_perpG05 * (By_perpG05' * Bx_perpG05)));
    v = reshape(v - (1 - 1 / sqrt(2)) * y * (y' * v), n * p, 1);
    output = (v - (Qx + Qy) * (Qy' * v));
    output = reshape(output, n, p);
    output = output + (sqrt(2) - 1) * x * (x' * output);
    TranvParams{1} = Qx;
    TranvParams{2} = Qy;
end

function output = TranH_smooth_Quo_NonIn(x, d, y, H)
    [n, p] = size(x);
    Bx_perpG05 = build_perp_bases_embedded(x);
    By_perpG05 = build_perp_bases_embedded(y);
    Qx = qf_thin(full(By_perpG05 - Bx_perpG05 * (Bx_perpG05' * By_perpG05)));
    Qy = qf_thin(full(Bx_perpG05 - By_perpG05 * (By_perpG05' * Bx_perpG05)));
    output = BGn05_quotient(x, H);
    output = G05B_quotient(x, output);
    output = output - output * (Qx + Qy) * Qy';
    output = output - (Qy + Qx) * (Qx' * output);
    output = BG05_quotient(y, output);
    output = Gn05B_quotient(y, output);
end

%% Quo Iso: intrinsic approach
function [output, TranvParams] = Tranv_smooth_Quo_intr(x, d, y, v, TranvParams)
    TranvParams = [];
    output = v;
end

function [output, TranvParams] = Tranv_smooth_Quo_intr_Params(x, d, y, v, TranvParams)
    output = v;
end

function [output, TranvParams] = invTranv_smooth_Quo_intr(x, d, y, v, TranvParams)
    TranvParams = [];
    output = v;
end

function output = TranH_smooth_Quo_intr(x, d, y, H)
    output = H;
end

%%assistant function------------------------------------------------------
function [output, R] = qf(B)
    k = size(B, 2);
    [output, R] = qr(B, 0);
    diagR = diag(R);
    diagR(diagR == 0) = 1;
    output = output * spdiags(sign(diagR), 0, k, k);
end

function [output, R] = qf_thin(B)
    k = size(B, 2);
    [output, R] = qr(B, 0);
    diagR = diag(R);
    output = output(:, abs(diagR) > 1e-10);
    output = output * spdiags(sign(diagR), 0, size(output, 2), size(output, 2));
end

function output = GB_quotient(x, B)
    [n, p] = size(x);
    output = zeros(size(B));
    for i = 1 : p
        output((i - 1) * n + 1 : i * n, :) = B((i - 1) * n + 1 : i * n, :) - (0.5 * x) * x'* B((i - 1) * n + 1 : i * n, :);
    end
end

function output = G05B_quotient(x, B)
    a = 1 - 1 / sqrt(2);
    [n, p] = size(x);
    output = zeros(size(B));
    for i = 1 : p
        output((i - 1) * n + 1 : i * n, :) = B((i - 1) * n + 1 : i * n, :) - (a * x) * x'* B((i - 1) * n + 1 : i * n, :);
    end
end

function output = Gn05B_quotient(x, B)
    a = sqrt(2) - 1;
    [n, p] = size(x);
    output = zeros(size(B));
    for i = 1 : p
        output((i - 1) * n + 1 : i * n, :) = B((i - 1) * n + 1 : i * n, :) + (a * x) * x'* B((i - 1) * n + 1 : i * n, :);
    end
end

function output = BG05_quotient(x, B)
    a = 1 - 1 / sqrt(2);
    [n, p] = size(x);
    output = zeros(size(B));
    for i = 1 : p
        output(:, (i - 1) * n + 1 : i * n) = B(:, (i - 1) * n + 1 : i * n) - a * (B(:, (i - 1) * n + 1 : i * n) * x) * x';
    end
end

function output = BGn05_quotient(x, B)
    a = sqrt(2) - 1;
    [n, p] = size(x);
    output = zeros(size(B));
    for i = 1 : p
        output(:, (i - 1) * n + 1 : i * n) = B(:, (i - 1) * n + 1 : i * n) + a * (B(:, (i - 1) * n + 1 : i * n) * x) * x';
    end
end

function output = build_perp_bases_embedded(x)
    r2 = sqrt(2);
    [n, p] = size(x);
    output = sparse(n * p, p * (p + 1) / 2);
%     output = zeros(n * p, p * (p + 1) / 2);
    times = 1;
    for j = 1 : p
        st = (j - 1) * n + 1;
        en = j * n;
        output(st : en, times) = x(:, j);
        times = times + 1;
    end

    for i = 2 : p
        for j = i : p
            st = (j - i) * n + 1;
            en = (j - i + 1) * n;
            output(st : en, times) = x(:, j) / r2;
            st = (j - 1) * n + 1;
            en = j * n;
            output(st : en, times) = x(:, j - i + 1) / r2;
            times = times + 1;
        end
    end
end

function output = build_perp_bases_quotient(x)
    r2 = sqrt(2);
    [n, p] = size(x);
    output = sparse(n * p, p * (p + 1) / 2);
%     output = zeros(n * p, p * (p + 1) / 2);
    times = 1;
    for j = 1 : p
        st = (j - 1) * n + 1;
        en = j * n;
        output(st : en, times) = x(:, j) * r2;
        times = times + 1;
    end

    for i = 2 : p
        for j = i : p
            st = (j - i) * n + 1;
            en = (j - i + 1) * n;
            output(st : en, times) = x(:, j);
            st = (j - 1) * n + 1;
            en = j * n;
            output(st : en, times) = x(:, j - i + 1);
            times = times + 1;
        end
    end
end

function output = build_bases_embedded(x)
    [n, p] = size(x);
    x_perp = null(x');
    x = x / sqrt(2);
    d = n * p - p * (p + 1) / 2;
    output = sparse(n * p, d);
%     output = zeros(n * p, d);
    col = 1;
    for i = 2 : p
        for j = i : p
            output((i - 2) * n + 1 : (i - 1) * n, col) = - x(:, j);
            output((j - 1) * n + 1 : j * n, col) = x(:, i - 1);
            col = col + 1;
        end
    end
    for i = 1 : p
        output((i - 1) * n + 1 : i * n, col : col + n - p - 1) = x_perp(1 : n, 1 : n - p);
        col = col + n - p;
    end
end

function output = build_bases_quotient(x)
    [n, p] = size(x);
    x_perp = null(x');
    d = n * p - p * (p + 1) / 2;
    output = sparse(n * p, d);
%     output = zeros(n * p, d);
    col = 1;
    for i = 2 : p
        for j = i : p
            output((i - 2) * n + 1 : (i - 1) * n, col) = - x(:, j);
            output((j - 1) * n + 1 : j * n, col) = x(:, i - 1);
            col = col + 1;
        end
    end
    for i = 1 : p
        output((i - 1) * n + 1 : i * n, col : col + n - p - 1) = x_perp(1 : n, 1 : n - p);
        col = col + n - p;
    end
end

function dw = ode_parallel_vt_embedded(t, w, M1, M2, A, n, p)
    w = reshape(w, n, p);
    Temp1 = expm(M1 * t);
    Temp2 = expm(- t * A);
    r = M2 * Temp1(:, 1 : p) * Temp2;
    dr = M2 * Temp1(:, p + 1 : 2 * p) * Temp2;
    dw = reshape(- 0.5 * r * (dr' * w + w' * dr), n * p, 1);
end

function dw = ode_parallel_vt_quotient(t, w, M1, M2, A, n, p)
    w = reshape(w, n, p);
    Temp = expm(M1 * t);
    r = M2 * Temp(:, 1 : p);
    dr = M2 * (Temp * M1(:, 1 : p));
    dw = reshape(- 0.5 * (dr * w' + w * dr') * r - 0.5 * r * (dr' * (w - r * r' * w) + w' * (dr - r * r' * dr)), n * p, 1);
end

function [t, y] = runge_kutta1(fhandle, y0, h, a, b)
    n = floor((b - a) / h);
    y = zeros(length(y0), n + 1);
    t = zeros(n + 1, 1);
    t(1) = a;
    y(:, 1) = y0;
    for i = 1 : n
        t(i + 1) = t(i) + h;
        k1 = fhandle(t(i), y(:, i));
        k2 = fhandle(t(i) + h / 2, y(:, i) + h * k1 / 2);
        k3 = fhandle(t(i) + h / 2, y(:, i) + h * k2 / 2);
        k4 = fhandle(t(i) + h, y(:, i) + h * k3);
        y(:, i + 1) = y(:, i) + h * (k1 + 2 * k2 + 2 * k3 + k4) / 6;
    end
end

function output = embedded_intr_c(x, eta)
    x1 = x{1};
    x2 = x{2};
    r2 = sqrt(2);
    [n, p] = size(x1);
    omega = x1' * eta; % omega
    K = x2' * eta; % K
    omega = - sqrt(2) * omega;
    output = zeros(n * p - 0.5 * p * (p + 1), 1);
    indx = find(tril(ones(size(omega)), -1));
    output(1 : 0.5 * p * (p - 1)) = omega(indx);
%     output(1 : 0.5 * p * (p - 1)) = nonzeros(tril(omega, -1));
    output(0.5 * p * (p - 1) + 1 : end) = reshape(K, [], 1);
end

function output = quotient_intr_c(x, eta)
    x1 = x{1};
    x2 = x{2};
    [n, p] = size(x1);
    omega = x1' * eta; % omega
    K = x2' * eta; % K
    omega = - omega;
    output = zeros(n * p - 0.5 * p * (p + 1), 1);
    indx = find(tril(ones(size(omega)), -1));
    output(1 : 0.5 * p * (p - 1)) = omega(indx);
%     output(1 : 0.5 * p * (p - 1)) = nonzeros(tril(omega, -1));
    output(0.5 * p * (p - 1) + 1 : end) = reshape(K, [], 1);
end

function output = embedded_full_c(x, v)
    x1 = x{1};
    x2 = x{2};
    r2 = sqrt(2);
    [n, p] = size(x1);
    omega = tril(ones(p, p), -1);
    indx = find(omega);
    omega(indx) = v(1 : 0.5 * p * (p - 1)) / r2;
    omega = omega' - omega;
    K = reshape(v(0.5 * p * (p - 1) + 1 : end), (n - p), p);
    output = x1 * omega + x2 * K;
end

function output = quotient_full_c(x, v)
    x1 = x{1};
    x2 = x{2};
    [n, p] = size(x1);
    omega = tril(ones(p, p), -1);
    indx = find(omega);
    omega(indx) = v(1 : 0.5 * p * (p - 1));
    omega = omega' - omega;
    K = reshape(v(0.5 * p * (p - 1) + 1 : end), (n - p), p);
    output = x1 * omega + x2 * K;
end

function output = mynull(A)
    [m, n] = size(A);
    randn('state', 1);
    T = randn(n, n);
    T(:, 1 : m) = A';
    V = qf(T);
    output = V(:,m+1:n);
end
